#' Function assembles normalization results from different chip types
#' into a single data frame. Note, this is possible since the three chip types
#' used by the CMap2 project contain nearly identical probe set ids. Typically,
#' the input for this step is generated by the upstream \code{normalizeCel} and
#' \code{combineResults} functions. For more context, please consult the vignette
#' of this package.
#' @title Combine Normalized Data from Different Chip Types
#' @param chiptype_dir character vector containing paths to chiptype directories
#' @param norm_method normalization method, one of "MAS5" or "rma".
#' @return Object of class \code{data.frame} containing normalization values
#' @examples 
#' # chiptype_dir <- unique(readRDS("./data/chiptype.rds"))
#' combineNormRes(chiptype_dir, norm_method="not run")
#' @export
combineNormRes <- function(chiptype_dir, norm_method) {
    if(norm_method=="MAS5"){
        df1 <- readRDS(paste0("data/", chiptype_dir[1], "/", "all_mas5exprs.rds"))
        df2 <- readRDS(paste0("data/", chiptype_dir[2], "/", "all_mas5exprs.rds"))
        df3 <- readRDS(paste0("data/", chiptype_dir[3], "/", "all_mas5exprs.rds"))
    } else if (norm_method=="rma"){
        df1 <- read.delim(paste0("./data/", chiptype_dir[1], "/rma_exprs.xls"), 
                          sep="\t", header=TRUE, row.names=1, check.names=FALSE)
        df2 <- read.delim(paste0("./data/", chiptype_dir[2], "/rma_exprs.xls"), 
                          sep="\t", header=TRUE, row.names=1, check.names=FALSE)
        df3 <- read.delim(paste0("./data/", chiptype_dir[3], "/rma_exprs.xls"), 
                          sep="\t", header=TRUE, row.names=1, check.names=FALSE)
    } else {
        message("Please set norm_method as one of 'MAS5' or 'rma'!")
        return(NULL)
    }
    affyid <- rownames(df1)[rownames(df1) %in% rownames(df2)]
    affyid <- affyid[affyid %in% rownames(df3)]
    normdf <- cbind(df1[affyid,], df2[affyid,], df3[affyid,])
    return(normdf)
}
