% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/motif_tree.R
\name{motif_tree}
\alias{motif_tree}
\title{Generate \pkg{ggplot2} motif trees with \pkg{ggtree}.}
\usage{
motif_tree(motifs, layout = "circular", linecol = "family",
  labels = "none", tipsize = "none", legend = TRUE,
  branch.length = "none", db.scores, method = "EUCL", use.type = "PPM",
  min.overlap = 6, min.position.ic = 0, tryRC = TRUE, min.mean.ic = 0,
  relative_entropy = FALSE, progress = FALSE, nthreads = 1,
  score.strat = "a.mean", ...)
}
\arguments{
\item{motifs}{\code{list}, \code{dist} See \code{\link[=convert_motifs]{convert_motifs()}} for
available formats. Alternatively, the resulting comparison matrix from
\code{\link[=compare_motifs]{compare_motifs()}} (run \code{as.dist(results)} beforehand; if the comparison was
performed with a similarity metric, make sure to convert to distances first).}

\item{layout}{\code{character(1)} One of \code{c('rectangular', 'slanted', 'fan', 'circular', 'radial', 'equal_angle', 'daylight')}. See \code{\link[ggtree:ggtree]{ggtree::ggtree()}}.}

\item{linecol}{\code{character(1)} \linkS4class{universalmotif} slot to use to
colour lines (e.g. 'family'). Not available for \code{dist} input (see examples
for how to add it manually). See \code{\link[ggtree:ggtree]{ggtree::ggtree()}}.}

\item{labels}{\code{character(1)} \linkS4class{universalmotif} slot to use to label
tips (e.g. 'name'). For \code{dist} input, only 'name' is available.
See \code{\link[ggtree:ggtree]{ggtree::ggtree()}}.}

\item{tipsize}{\code{character(1)} \linkS4class{universalmotif} slot to use to
control tip size (e.g. 'icscore'). Not available for \code{dist} input (see
examples for how to add it manually). See \code{\link[ggtree:ggtree]{ggtree::ggtree()}}.}

\item{legend}{\code{logical(1)} Show legend for line colour and tip size.
See \code{\link[ggtree:ggtree]{ggtree::ggtree()}}.}

\item{branch.length}{\code{character(1)} If 'none', draw a cladogram.
See \code{\link[ggtree:ggtree]{ggtree::ggtree()}}.}

\item{db.scores}{\code{data.frame} See \code{\link[=compare_motifs]{compare_motifs()}}.}

\item{method}{\code{character(1)} One of PCC, EUCL, SW, KL, ALLR, BHAT, HELL,
SEUCL, MAN, ALLR_LL, WEUCL, WPCC. See details.}

\item{use.type}{\code{character(1)}c('PPM', 'ICM')\verb{. The latter allows for taking into account the background frequencies (only if }relative_entropy = TRUE`). See \code{\link[=compare_motifs]{compare_motifs()}}.}

\item{min.overlap}{\code{numeric(1)} Minimum overlap required when aligning the
motifs. Setting this to a number higher then the width of the motifs
will not allow any overhangs. Can also be a number between 0 and 1,
representing the minimum fraction that the motifs must overlap.}

\item{min.position.ic}{\code{numeric(1)} Minimum information content required between
individual alignment positions for it to be counted in the final alignment
score. It is recommended to use this together with \code{normalise.scores = TRUE},
as this will help punish scores resulting from only a fraction of an
alignment.}

\item{tryRC}{\code{logical(1)} Try the reverse complement of the motifs as well,
report the best score.}

\item{min.mean.ic}{\code{numeric(1)} Minimum mean information content between the
two motifs for an alignment to be scored. This helps prevent scoring
alignments between low information content regions of two motifs. Note that
this can result in some comparisons failing if no alignment passes the
mean IC threshold. Use \code{\link[=average_ic]{average_ic()}} to filter out low IC motifs to get around
this if you want to avoid getting \code{NA}s in your output.}

\item{relative_entropy}{\code{logical(1)} Change the ICM calculation affecting
\code{min.position.ic} and \code{min.mean.ic}. See \code{\link[=convert_type]{convert_type()}}.}

\item{progress}{\code{logical(1)} Show message regarding current step.}

\item{nthreads}{\code{numeric(1)} Run \code{\link[=compare_motifs]{compare_motifs()}} in parallel with \code{nthreads}
threads. \code{nthreads = 0} uses all available threads.}

\item{score.strat}{\code{character(1)} How to handle column scores calculated from
motif alignments. "sum": add up all scores. "a.mean": take the arithmetic
mean. "g.mean": take the geometric mean. "median": take the median.
"wa.mean", "wg.mean": weighted arithmetic/geometric mean. "fzt": Fisher
Z-transform. Weights are the
total information content shared between aligned columns.}

\item{...}{\pkg{ggtree} params. See \code{\link[ggtree:ggtree]{ggtree::ggtree()}}.}
}
\value{
ggplot object.
}
\description{
For more powerful motif tree functions, see the \pkg{motifStack} package.
The \code{\link[=motif_tree]{motif_tree()}} function compares motifs with \code{\link[=compare_motifs]{compare_motifs()}} to create
a distance matrix, which is used to generate a phylogeny.
This can be plotted with \code{\link[ggtree:ggtree]{ggtree::ggtree()}}. The purpose of this function
is simply to combine the \code{\link[=compare_motifs]{compare_motifs()}} and \code{\link[ggtree:ggtree]{ggtree::ggtree()}} steps
into one. For more control over tree creation, it is recommend to do these
steps separately. See the "Motif comparisons and P-values" vignette for such
a workthrough. This function requires the \pkg{ape} and \pkg{ggtree} packages
to be installed separately.
}
\details{
See \code{\link[=compare_motifs]{compare_motifs()}} for more info on comparison parameters.
}
\examples{
jaspar <- read_jaspar(system.file("extdata", "jaspar.txt",
                                  package = "universalmotif"))
if (requireNamespace("ggtree", quietly = TRUE)) {
jaspar.tree <- motif_tree(jaspar, linecol = "none", labels = "name",
                          layout = "rectangular")
}

\dontrun{
## When inputting a dist object, the linecol and tipsize options are
## not available. To add these manually:

library(MotifDb)
library(ggtree)
library(ggplot2)

motifs <- filter_motifs(MotifDb, organism = "Athaliana")[1:50]
comparison <- compare_motifs(motifs, method = "PCC", score.strat = "a.mean")
comparison <- as.dist(1 - comparison)
mot.names <- attr(comparison, "Labels")
tree <- motif_tree(comparison)

annotations <- data.frame(label = mot.names,
                          icscore = sapply(motifs, function(x) x["icscore"]),
                          family = sapply(motifs, function(x) x["family"]))

tree <- tree \%<+\% annotations +
          geom_tippoint(aes(size = icscore)) +
          aes(colour = family) +
          theme(legend.position = "right",
                legend.title = element_blank())
}

}
\references{
Wickham H (2009). \emph{ggplot2: Elegant Graphics for Data Analysis}.
Springer-Verlag New York. ISBN 978-0-387-98140-6, <URL:
http://ggplot2.org>.

Yu G, Smith D, Zhu H, Guan Y, Lam TT (2017). “ggtree: an R package
for visualization and annotation of phylogenetic trees with their
covariates and other associated data.” \emph{Methods in Ecology and
Evolution}, \strong{8}, 28-36. doi: 10.1111/2041-210X.12628.
}
\seealso{
\code{\link[motifStack:motifStack]{motifStack::motifStack()}}, \code{\link[=compare_motifs]{compare_motifs()}},
\code{\link[ggtree:ggtree]{ggtree::ggtree()}}, \code{\link[ggplot2:ggplot]{ggplot2::ggplot()}}
}
\author{
Benjamin Jean-Marie Tremblay, \email{benjamin.tremblay@uwaterloo.ca}
}
