#' Heatmap of all plates
#'
#' Put all individual plates in one graph, values are the readout in experiments.
#'
#' @param dta synthetic lethal RNAi screen data
#' @param base_size basic font size used for x/y axis and title for heatmaps
#' @param heatmap_col color function generated by colorRampPalette.
#' @return a ggplot object
#' @examples
#' data(example_dt)
#' plateHeatmap(example_dt)
#' @export
plateHeatmap <- function(dta, base_size = 12, heatmap_col = NULL) {
  COL_NAME <- new_name <- PLATE <- EXPERIMENT_MODIFICATION <- ROW_NAME <- value <- NULL

  dta_2 <- copy(dta) %>%
    .[, new_name := paste(PLATE, EXPERIMENT_MODIFICATION, sep = " ")] %>%
    melt(id.vars = c("ROW_NAME", "COL_NAME", "new_name"), measure.vars = "READOUT") %>%
    .[, ROW_NAME := factor(ROW_NAME, levels = unique(ROW_NAME) %>% sort %>% rev)] %>%
    .[, COL_NAME := factor(COL_NAME, levels = unique(COL_NAME) %>% as.numeric %>% sort %>% rev %>% as.character)]

  if (is.null(heatmap_col)) heatmap_col <- colorRampPalette(rev(RColorBrewer::brewer.pal(11, "Spectral")))

  ggplot(as.data.frame(dta_2), aes(COL_NAME, ROW_NAME)) +
    geom_tile(aes(fill = value), colour = "white") +
    scale_fill_gradientn(colours = heatmap_col(100)) +
    theme_bw() +
    facet_wrap(~ new_name) +
    theme(axis.text.x = element_text(size = base_size * 0.5),
          axis.text.y = element_text(size = base_size * 0.5),
          plot.title  = element_text(size = base_size * 0.8),
          panel.grid.major = element_blank(),
          panel.grid.minor = element_blank(),
          panel.background = element_rect(colour = "black", size = 1)) +
    labs(x = "", y = "") +
    coord_equal() +
    #- Expand not working.
    # scale_x_continuous(expand = c(0, 0)) +
    theme(panel.background = element_rect(fill = "white", colour = "white"))
}
