% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/startracDiversity.R
\name{StartracDiversity}
\alias{StartracDiversity}
\title{Calculate Startrac-based Diversity Indices}
\usage{
StartracDiversity(
  sc.data,
  cloneCall = "strict",
  chain = "both",
  index = c("expa", "migr", "tran"),
  type = NULL,
  group.by = NULL,
  pairwise = NULL,
  exportTable = FALSE,
  palette = "inferno",
  ...
)
}
\arguments{
\item{sc.data}{The single-cell object after \code{\link[=combineExpression]{combineExpression()}}.
For SCE objects, the cluster variable must be in the meta data under
"cluster".}

\item{cloneCall}{Defines the clonal sequence grouping. Accepted values
are: \code{gene} (VDJC genes), \code{nt} (CDR3 nucleotide sequence), \code{aa} (CDR3 amino
acid sequence), or \code{strict} (VDJC + nt). A custom column header can also be used.}

\item{chain}{The TCR/BCR chain to use. Use \code{both} to include both chains
(e.g., TRA/TRB). Accepted values: \code{TRA}, \code{TRB}, \code{TRG}, \code{TRD}, \code{IGH}, \code{IGL}
(for both light chains), \code{both}.}

\item{index}{A character vector specifying which indices to calculate.
Options: "expa", "migr", "tran". Default is all three.}

\item{type}{The metadata variable that specifies tissue type for migration
analysis.}

\item{group.by}{A column header in the metadata or lists to group the analysis
by (e.g., "sample", "treatment"). If \code{NULL}, data will be analyzed as
by list element or active identity in the case of single-cell objects.}

\item{pairwise}{The metadata column to be used for pairwise comparisons.
Set to the \code{type} variable for pairwise migration or "cluster" for
pairwise transition.}

\item{exportTable}{If \code{TRUE}, returns a data frame or matrix of the results
instead of a plot.}

\item{palette}{Colors to use in visualization - input any
\link[grDevices:palettes]{hcl.pals}.}

\item{...}{Additional arguments passed to the ggplot theme}
}
\value{
A ggplot object visualizing STARTRAC diversity metrics or data.frame if
\code{exportTable = TRUE}.
}
\description{
This function utilizes the STARTRAC approach to calculate T cell
diversity metrics based on the work of Zhang et al. (2018, Nature)
\href{https://pubmed.ncbi.nlm.nih.gov/30479382/}{PMID: 30479382}. It can compute
three distinct indices: clonal expansion (\code{expa}), cross-tissue migration
(\code{migr}), and state transition (\code{tran}).
}
\details{
The function requires a \code{type} variable in the metadata, which specifies the
tissue origin or any other categorical variable for migration analysis.

\strong{Indices:}
\itemize{
\item{\strong{expa (Clonal Expansion):}} Measures the extent of clonal
proliferation within a T cell cluster. It is calculated as
\verb{1 - normalized Shannon entropy}. A higher value indicates greater
expansion of a few clones.
\item{\strong{migr (Cross-Tissue Migration):}} Quantifies the movement of
clonal T cells across different tissues (as defined by the \code{type}
parameter). It is based on the entropy of a clonotype's distribution
across tissues.
\item{\strong{tran (State Transition):}} Measures the developmental
transition of clonal T cells between different functional clusters.
It is based on the entropy  of a clonotype's distribution across
clusters.
}

\strong{Pairwise Analysis:}
The \code{pairwise} parameter enables the calculation of migration or transition
between specific pairs of tissues or clusters, respectively.
\itemize{
\item{For migration (\code{index = "migr"}), set \code{pairwise} to the \code{type} column
(e.g., \code{pairwise = "Type"}).}
\item{For transition (\code{index = "tran"}), set \code{pairwise} to \code{"cluster"}.}
}
}
\examples{
# Getting the combined contigs
combined <- combineTCR(contig_list, 
                        samples = c("P17B", "P17L", "P18B", "P18L", 
                                    "P19B","P19L", "P20B", "P20L"))

# Getting a sample of a Seurat object
scRep_example  <- get(data("scRep_example"))
scRep_example  <- combineExpression(combined, scRep_example)
scRep_example$Patient <- substring(scRep_example$orig.ident,1,3)
scRep_example$Type <- substring(scRep_example$orig.ident,4,4) 

# Calculate a single index (expansion)
StartracDiversity(scRep_example, 
                  type = "Type", 
                  group.by = "Patient",
                  index = "expa")
                  
# Calculate pairwise transition 
StartracDiversity(scRep_example, 
                  type = "Type", 
                  group.by = "Patient",
                  index = "tran",
                  pairwise = "cluster") 

}
\author{
Liangtao Zheng
}
\concept{SC_Functions}
