% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/edgeReport.R
\name{edgeReport}
\alias{edgeReport}
\title{Generate a HTML/PDF report exploring edgeR results}
\usage{
edgeReport(
  dge,
  object,
  project = "",
  intgroup,
  colors = NULL,
  pAdjustMethod = "BH",
  alpha = 0.1,
  independentFiltering = FALSE,
  filter,
  theta,
  filterFun,
  nBest = 500,
  nBestFeatures = 20,
  customCode = NULL,
  outdir = "edgeRexploration",
  output = "edgeRexploration",
  browse = interactive(),
  device = "png",
  template = NULL,
  searchURL = "http://www.ncbi.nlm.nih.gov/gene/?term=",
  theme = NULL,
  digits = 2,
  ...
)
}
\arguments{
\item{dge}{A \link[edgeR]{DGEList} object.}

\item{object}{A \link[edgeR:DGEList-class]{DGEExact} or
\link[edgeR:DGELRT-class]{DGELRT} object that contains p-values stored in
\code{object$table$PValue}.}

\item{project}{The title of the project.}

\item{intgroup}{interesting groups: a character vector of names in
\code{colData(x)} to use for grouping. This parameter is passed to functions
such as \link[DESeq2]{plotPCA}.}

\item{colors}{vector of colors used in heatmap. If \code{NULL}, then a
a default set of colors will be used. This argument is passed to
\link[pheatmap]{pheatmap}.}

\item{pAdjustMethod}{the method to use for adjusting p-values, see
\link[stats]{p.adjust}. This argument will be passed to
\link[DESeq2]{results}.}

\item{alpha}{the significance cutoff used for optimizing the independent
filtering (by default 0.1). If the adjusted p-value cutoff (FDR) will be a
value other than 0.1, alpha should be set to that value. This argument will
be passed to \link[DESeq2]{results}.}

\item{independentFiltering}{logical, whether independent filtering should be
applied automatically. By default it's set to \code{FALSE} in contrast with
the default used in \link[DESeq2]{results} to match \code{edgeR}'s behavior.}

\item{filter}{the vector of filter statistics over which the independent filtering
will be optimized. By default the logCPM will be used if
\code{independentFiltering} is set to \code{TRUE}. It can also be a length
1 character vector specifying one of the column names of \code{object$table}.}

\item{theta}{the quantiles at which to assess the number of rejections from independent filtering. This argument is passed \link[DESeq2]{results}.}

\item{filterFun}{an optional custom function as described in
\link[DESeq2]{results}.}

\item{nBest}{The number of features to include in the interactive
table. Features are ordered by their adjusted p-values.}

\item{nBestFeatures}{The number of best features to make plots of their
counts. We recommend a small number, say 20.}

\item{customCode}{An absolute path to a child R Markdown file with code to be
evaluated before the reproducibility section. Its useful for users who want
to customize the report by adding conclusions derived from the data and/or
further quality checks and plots.}

\item{outdir}{The name of output directory.}

\item{output}{The name of output HTML file (without the html extension).}

\item{browse}{If \code{TRUE} the HTML report is opened in your browser once
it's completed.}

\item{device}{The graphical device used when knitting. See more at
\url{http://yihui.name/knitr/options} (\code{dev} argument).}

\item{template}{Template file to use for the report. If not provided, will
use the default file found in DESeq2Exploration/DESeq2Exploration.Rmd
within the package source.}

\item{searchURL}{A url used for searching the name of the features in
the web. By default \code{http://www.ncbi.nlm.nih.gov/gene/?term=} is used
which is the recommended option when features are genes. It's only used
when the output is a HTML file.}

\item{theme}{A ggplot2 \link[ggplot2]{theme} to use for the plots made with
ggplot2.}

\item{digits}{The number of digits to round to in the interactive table of
the top \code{nBestFeatures}. Note that p-values and adjusted p-values won't
be rounded.}

\item{...}{Arguments passed to other methods and/or advanced arguments.
Advanced arguments:
\describe{
\item{software }{ The name of the package used for performing the
differential expression analysis. Either \code{DESeq2} or \code{edgeR}.}
\item{dge }{ A \link[edgeR]{DGEList} object. \code{NULL} by default and only
used by \link{edgeReport}.}
\item{theCall }{ The function call. \code{NULL} by default and only used by
\link{edgeReport}.}
\item{output_format }{ Either \code{html_document}, \code{pdf_document} or
\code{knitrBootstrap::bootstrap_document} unless you modify the YAML
template.}
\item{clean }{ Logical, whether to clean the results or not. Passed to
\link[rmarkdown]{render}.}
}}
}
\value{
An HTML report with a basic exploration for the given set of edgeR
results. See the example report at
\url{http://leekgroup.github.io/regionReport/reference/edgeReport-example/edgeRexploration.html}.
}
\description{
This function generates a HTML report with exploratory data analysis plots
for edgeR results created. Other output formats are possible such as PDF
reports but they lose the interactivity. Users can easily append
to the report by providing a R Markdown file to \code{customCode}, or can
customize the entire template by providing an R Markdown file to
\code{template}.
}
\details{
Set \code{output_format} to \code{'knitrBootstrap::bootstrap_document'} or
\code{'pdf_document'} if you want a HTML report styled by knitrBootstrap or
a PDF report respectively. If using knitrBootstrap, we recommend the version
available only via GitHub at \url{https://github.com/jimhester/knitrBootstrap}
which has nicer features than the current version available via CRAN.

If you modify the YAML front matter of \code{template}, you can use other
values for \code{output_format}.

This report is similar to the one created by \link{DESeq2Report} with two
additional plots exclusive for edgeR results. We designed the reports to be
very similar intentionally and use the Bioconductor package DEFormats to
achieve this goal.
}
\examples{

## Create example data using DEFormats
library("DEFormats")
set.seed(20160407)
counts <- simulateRnaSeqData()
group <- rep(c("A", "B"), each = 3)

## Create DGEList object
library("edgeR")
dge <- DGEList(counts, group = group)

## Perform DE analysis with edgeR
design <- model.matrix(~group)
dge <- estimateDisp(dge, design)
fit <- glmFit(dge, design)
lrt <- glmLRT(fit, coef = 2)

## The output will be saved in the 'edgeReport-example' directory
dir.create("edgeReport-example", showWarnings = FALSE, recursive = TRUE)

## Generate the HTML report
report <- edgeReport(dge, lrt,
    project = "edgeR-example", intgroup = "group",
    outdir = "edgeReport-example"
)

if (interactive()) {
    ## Browse the report
    browseURL(report)
}

## See the example report at
## http://leekgroup.github.io/regionReport/reference/edgeReport-example/edgeRexploration.html
\dontrun{
## Note that you can run the example using:
example("edgeReport", "regionReport", ask = FALSE)
}

}
\author{
Leonardo Collado-Torres
}
