% Generated by roxygen2 (4.0.0): do not edit by hand
\name{slidingMean}
\alias{slidingMean}
\title{Data smoothing for peptide microarray.}
\usage{
slidingMean(peptideSet, width = 9, verbose = FALSE, split.by.clade = TRUE)
}
\arguments{
\item{peptideSet}{A \code{peptideSet}. The expression data for the peptides as
well as annotations and ranges. The range information is required to run this function.}

\item{width}{A \code{numeric}. The width of the sliding window.}

\item{verbose}{A \code{logical}. If set to TRUE, progress information will be displayed.}

\item{split.by.clade}{A \code{logical}. If TRUE, the peptides will be smoothed by
clade (See details section below for more information).}
}
\value{
A \code{peptideSet} object with smoothed intensities.
}
\description{
This function applies a sliding mean window to intensities to reduce noise
generated by experimental variation, as well as take advantage of the overlapping
nature of array peptides to share signal.
}
\details{
Peptide membership in the sliding mean window is determined by its position and
the width argument. Two peptides are in the same window if the difference in their
positions is less than or equal to width/2. A peptide's position is taken to be
position(peptideSet).

A peptide's intensity is replaced by the mean of all peptide intensities within
the peptide's sliding mean window.

When split.by.clade = TRUE, peptides are smoothed within clades defined by the
clade column of the GRanges object occupying the featureRange slot of
peptideSet. If set to FALSE, a peptide at a given position will borrow
information from the neighboring peptides as well as the ones from other
clades around this position.
}
\examples{
## This example curated from the vignette -- please see vignette("pepStat")
## for more information
if (require("pepDat")) {

  ## Get example GPR files + associated mapping file
  dirToParse <- system.file("extdata/gpr_samples", package = "pepDat")
  mapFile <- system.file("extdata/mapping.csv", package = "pepDat")

  ## Make a peptide set
  pSet <- makePeptideSet(files = NULL, path = dirToParse,
                         mapping.file = mapFile, log=TRUE)

  ## Plot array images -- useful for quality control
  plotArrayImage(pSet, array.index = 1)
  plotArrayResiduals(pSet, array.index = 1, smooth = TRUE)

  ## Summarize peptides, using pep_hxb2 as the position database
  data(pep_hxb2)
  psSet <- summarizePeptides(pSet, summary = "mean", position = pep_hxb2)

  ## Normalize the peptide set
  pnSet <- normalizeArray(psSet)

  ## Smooth
  psmSet <- slidingMean(pnSet, width = 9)

  ## Make calls
  calls <- makeCalls(psmSet, freq = TRUE, group = "treatment",
                     cutoff = .1, method = "FDR", verbose = TRUE)

  ## Produce a summary of the results
  summary <- restab(psmSet, calls)

}
}
\author{
Gregory Imholte
}
\seealso{
\code{\link{summarizePeptides}}, \code{\link{normalizeArray}}
}

