% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/primers.R, R/PrimerEvaluation.R,
%   R/constraints_eval.R, R/filters.R, R/optimization_ILP.R,
%   R/primer_significance.R
\name{PrimerEval}
\alias{PrimerEval}
\alias{check_restriction_sites}
\alias{check_constraints}
\alias{filter_primers}
\alias{subset_primer_set}
\alias{primer_significance}
\title{Primer Evaluation.}
\usage{
check_restriction_sites(
  primer.df,
  template.df,
  adapter.action = c("warn", "rm"),
  selected = NULL,
  only.confident.calls = TRUE,
  updateProgress = NULL
)

check_constraints(
  primer.df,
  template.df,
  settings,
  active.constraints = names(constraints(settings)),
  to.compute.constraints = active.constraints,
  for.shiny = FALSE,
  updateProgress = NULL
)

filter_primers(
  primer.df,
  template.df,
  settings,
  active.constraints = names(constraints(settings))
)

subset_primer_set(
  primer.df,
  template.df,
  k = 1,
  groups = NULL,
  identifier = NULL,
  cur.results.loc = NULL
)

primer_significance(primer.df, set.name = NULL, active.constraints = NULL)
}
\arguments{
\item{primer.df}{A \code{Primers} object containing the primers
whose properties are to be checked.}

\item{template.df}{A \code{Templates} object containing the 
template sequences corresponding to \code{primer.df}.}

\item{adapter.action}{The action to be performed when adapter sequences
are found. Either "warn" to issue a warning about adapter sequences or
"rm" to remove identified adapter sequences. Currently, only
the default setting ("warn") is supported.}

\item{selected}{Names of restriction sites that are to be checked.
By default \code{selected} is \code{NULL} in which case all REBASE 
restriction sites are taken into account.}

\item{only.confident.calls}{Whether only confident calls
of restriction sites are returned.
All restriction site call is considered \emph{confident} if the restriction site
is located in a region that does not match the template sequences.
Note that this classification requires that the provided primers
are somehow complementary to the provided templates.
In contrast, non-confident restriction site calls are 
based solely on the primer sequences and do not take the templates
into account, resulting in more false positive calls of restriction sites.}

\item{updateProgress}{Progress callback function for shiny. The defaut is
\code{NULL} meaning that no progress is monitored via the Shiny interface.}

\item{settings}{A \code{DesignSettings} object containing the 
constraints that are to be considered.}

\item{active.constraints}{A subset of the constraint identifiers 
provided by \code{settings} that are to be checked
for fulfillment. By default \code{active.constraints} is \code{NULL} such that
all constraints found in \code{settings} are evaluated. Otherwise,
only the constraints specified via \code{active.constraints} 
that are available in \code{settings} are considered.}

\item{to.compute.constraints}{Constraints that are to be computed.
By default, \code{to.compute.constraints} is set to \code{NULL} such that
all \code{active.constraints} are computed. If \code{to.compute.constraints}
is a subset of \code{active.constraints}, all constraints specified
via \code{active.constraints} are evaluated for fulfillment,
but only the constraints in \code{to.compute.constraints} are newly calculated.}

\item{for.shiny}{Whether the output of the function shall be
formatted as HTML. The default setting is \code{FALSE}.}

\item{k}{The spacing between generated primer subset sizes. By default,
\code{k} is set to 1 such that all primer subsets are constructed.}

\item{groups}{The identifiers of template groups according to which 
coverage should be determined. By default, \code{groups} is set to 
\code{NULL} such that all all covered templates are considered.}

\item{identifier}{An identifier for storing the primer set. By default,
\code{identifier} is set to  \code{NULL}.}

\item{cur.results.loc}{Directory for storing the results. By default,
\code{cur.results.loc} is set to \code{NULL}, which means that
no results are stored.}

\item{set.name}{An identifier for the input primers. If \code{NULL},
the run identifier is used.}
}
\value{
\code{check_restriction_sites} returns a data frame 
with possible restriction sites found in the primers.

\code{check_constraints} returns a \code{Primers} object 
that is augmented with columns providing the results for the evaluated 
constraints.
The \code{constraints_passed} column indicates whether all 
\code{active.constraints} were fulfilled.
The \code{EVAL_*} columns indicate the fulfillment of primer-specific constraints.
The \code{T_EVAL_*} columns indicate the fulfillment of template-specific
(e.g. coverage-based) constraints.
For the coverage computations, columns prefixed by \code{Basic_},
indicate the results from string matching, while all other results
(e.g. \code{primer_coverage}) indicate the expected coverage
after applying the coverage constraints specified in \code{settings}.
Columns prefixed by \code{Off_} indicate off-target binding results.

\code{filter_primers} returns a \code{Primers} object containing
only those primers fulfilling all specified constraints.

\code{subset_primer_set} returns a list with optimal primer subsets,
each of class \code{Primers}.

\code{primer_significance} returns a numeric providing
the p-value of the primer set according to Fisher's exact test.
The returned value has the following attributes: 
\describe{
\item{\code{test}}{The results of the significance test}
\item{\code{tab}}{The confusion matrix for Fisher's exact test}
\item{\code{constraints}}{The names of the considered constraints}
}
}
\description{
\describe{
\item{\code{check_constraints}}{Determines whether a set of primers
fulfills the constraints on the properties of the primers.}
\item{\code{check_restriction_sites}}{Checks a set of primers 
for the presence of restriction sites. 
To reduce the number of possible restriction sites,
only unambiguous restriction sites are taken into account and 
only common (typically used) restriction sites are checked if a common
restriction site can be found in a sequence.}
\item{\code{filter_primers}}{Filters a primer set according to 
the specified constraints such that all primers
that do not fulfill the constraints are removed from the primer set.}
\item{\code{primer_significance}}{Uses Fisher's exact test to determine 
the significance of a primer set according to its ratio of fulfilled 
constraints.}
\item{\code{subset_primer_set}}{Determines subsets of the input primer set
that are optimal with regard to the number of covered template sequences.}
}
}
\details{
When the optional argument
\code{active.constraints} is supplied to \code{check_constraints}, 
only a subset of the constraints
provided in \code{settings} is evaluated. Only constraints that
are defined in \code{settings} can be computed. For a detailed
description of all possible constraints and their options, please
consider the \code{\link{ConstraintSettings}} documentation.

\code{subset_primer_set} determines optimal subsets of the input primer set
by solving an integer-linear program.
Since the quality of the primers (in terms of properties) is not taken into
account when creating the subsets, this method should only be used
for primer sets that are already of high quality.

\code{primer_significance} computes the significance by comparing
the total count of fulfilled and failed constraints
with the corresponding counts of primer sets from the literature.
Significant p-values indicate primer sets whose rate of constraint 
fulfillment is higher compared to the reference sets.
}
\note{
Please note that some constraint computations 
may require the installation of additional programs; for more information
please view the documentation of \code{\link{DesignSettings}}.
}
\examples{

data(Ippolito)
# Check the first primer for restriction sites with respect to the first 10 templates
site.df <- check_restriction_sites(primer.df[1,], template.df[1:10])

data(Ippolito)
settings.xml <- system.file("extdata", "settings", 
                 "C_Taq_PCR_high_stringency.xml", package = "openPrimeR")
settings <- read_settings(settings.xml)
# Check GC clamp and number of runs for all primers:
constraint.df <- check_constraints(primer.df, template.df, 
                     settings, active.constraints = c("gc_clamp", "no_runs"))
# Summarize the evaluation results
summary(constraint.df)

data(Ippolito)
filename <- system.file("extdata", "settings", 
             "C_Taq_PCR_high_stringency.xml", package = "openPrimeR")
settings <- read_settings(filename)
# Only retain the primers fulfilling the GC clamp constraint:
filtered.df <- filter_primers(primer.df, template.df, settings,
                 active.constraints = c("gc_ratio"))

# Determine optimal primer subsets
data(Ippolito)
primer.subsets <- subset_primer_set(primer.df, template.df, k = 3)

# Determine the significance of a primer set
data(Ippolito)
p.data <- primer_significance(primer.df, "Ippolito")
attr(p.data,"tab") # the confusion matrix
attr(p.data, "test") # results from Fisher's test
attr(p.data, "constraints") # considered constraints for the test
}
\references{
Roberts, R.J., Vincze, T., Posfai, J., Macelis, D. (2010) REBASE–a database for DNA restriction
and modification: enzymes, genes and genomes. Nucl. Acids Res. 38: D234-D236. http://rebase.neb.com
}
