% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/PlotScores.R
\name{PlotScores}
\alias{PlotScores}
\title{Plot gene signature scores using various methods.}
\usage{
PlotScores(
  data,
  metadata,
  gene_sets,
  method = c("ssGSEA", "logmedian", "ranking", "all"),
  ColorVariable = NULL,
  Variable = NULL,
  ColorValues = NULL,
  ConnectGroups = FALSE,
  ncol = NULL,
  nrow = NULL,
  title = NULL,
  widthTitle = 20,
  titlesize = 12,
  limits = NULL,
  legend_nrow = NULL,
  pointSize = 4,
  xlab = NULL,
  labsize = 10,
  compute_cohen = TRUE,
  cond_cohend = NULL,
  pvalcalc = FALSE,
  mode = c("simple", "medium", "extensive"),
  widthlegend = 22,
  sig_threshold = 0.05,
  cohen_threshold = 0.5,
  colorPalette = "Set3",
  cor = c("pearson", "spearman", "kendall")
)
}
\arguments{
\item{data}{A data frame of Normalised (non-transformed) counts where each
row is a gene and each column is a sample. Row names should contain gene
names, and column names should contain sample identifiers. \strong{(Required)}}

\item{metadata}{A data frame with sample-level attributes. Each row
corresponds to a sample, with the first column containing sample IDs that
match \code{colnames(data)}. \strong{Required if \code{method = "all"} or if
metadata-derived groupings or colors are used.}}

\item{gene_sets}{A named list of gene sets to score. For unidirectional gene
sets, provide a list of character vectors. For bidirectional gene sets,
provide a list of data frames with two columns: gene names and direction
(1 = up, -1 = down). \strong{(Required)}}

\item{method}{Scoring method to use. One of \code{"ssGSEA"}, \code{"logmedian"},
\code{"ranking"}, or \code{"all"} (default = \code{"logmedian"}). The \code{"all"} option
triggers a full analysis returning both heatmap and volcano plots. Other
values return single-score plots depending on \code{Variable} type.}

\item{ColorVariable}{Name of a metadata column to color points by. Used in
\strong{single-method mode} (\code{"ssGSEA"}, etc.). Ignored in \code{"all"} mode.}

\item{Variable}{Metadata column to define groups or numeric comparisons.
This is \strong{required if \code{method = "all"}} (used to compute and compare effect
sizes). If \code{NULL} and \code{method != "all"}, density plots of each signature
score across samples are shown (no grouping or comparison).}

\item{ColorValues}{Optional. A named vector or list of colors used to control
the coloring of plot elements across different methods and variable types.
Behavior depends on the combination of \code{method} and \code{Variable}.

If \code{method != "all"}, then:
\itemize{
\item If \code{Variable} is \code{NULL}, a single color will be applied in density plots
(default: \code{"#ECBD78"}).
\item If \code{Variable} is categorical, a named vector should map each level of
\code{Variable} (or \code{ColorVariable}) to a specific color. This overrides the
palette specified by \code{colorPalette}.
\item If \code{Variable} is numeric, a single color is applied to all points in the
scatter plot (default: \code{"#5264B6"}).
}

If \code{method == "all"}, then:
\itemize{
\item \code{ColorValues} can be a named list with two elements:
\itemize{
\item \code{heatmap}: a vector of two colors used as a diverging scale for the heatmap
of effect sizes (default: \code{c("#F9F4AE", "#B44141")}).
\item \code{volcano}: a named vector of colors used for labeling or grouping gene
signatures (e.g., in the volcano plot).
}
}

If not provided, defaults will be used for both components.

In all cases, \code{ColorValues} takes precedence over the default
\code{colorPalette} setting if specified.}

\item{ConnectGroups}{Logical. If \code{TRUE}, connects points by sample ID across
conditions (used for categorical variables and \code{method != "all"}).}

\item{ncol}{Number of columns for facet layout (used in both heatmaps and
score plots).}

\item{nrow}{Number of rows for facet layout (used in both heatmaps and score
plots).}

\item{title}{Plot title (optional).}

\item{widthTitle}{Width allocated for title (affects alignment).}

\item{titlesize}{Font size for plot title.}

\item{limits}{Y-axis limits (numeric vector of length 2).}

\item{legend_nrow}{Number of rows for plot legend (used in single-method
plots).}

\item{pointSize}{Numeric. Size of points in \strong{score plots} (violin or
scatter), used when plotting individual sample scores for both categorical
and numeric variables, including when \code{method = "all"}.}

\item{xlab}{Label for x-axis (optional; defaults to \code{Variable}).}

\item{labsize}{Font size for axis and facet labels.}

\item{compute_cohen}{Logical. Whether to compute Cohen's effect sizes in
\strong{score plots} (\code{method != "all"}). This only applies when \code{method != "all"}; ignored otherwise.
If the variable is categorical and \code{cond_cohend} is specified,
computes \strong{Cohen's d} for the specified comparison.
If the variable is categorical and \code{cond_cohend} is not specified, it computes
\strong{Cohen's d} if there are exactly two groups, or \strong{Cohen's f} if there are more than two groups.
If the variable is numeric, computes \strong{Cohen's f} regardless of \code{cond_cohend}.}

\item{cond_cohend}{Optional. List of length 2 with the two groups being used
to compute effect size. The values in each entry should be levels of
\code{Variable}. Used with \code{compute_cohen = TRUE}.}

\item{pvalcalc}{Logical. If \code{TRUE}, computes p-values between groups.}

\item{mode}{A string specifying the contrast mode when \code{method = "all"}.
Determines the complexity and breadth of comparisons performed between
group levels. Options are:

\code{"simple"} performs the minimal number of pairwise comparisons between
individual group levels (e.g., A - B, A - C). This is the default.

\code{"medium"} includes comparisons between one group and the union of all
other groups (e.g., A - (B + C + D)), enabling broader contrasts beyond simple pairs.

\code{"extensive"} allows for all possible algebraic combinations of group
levels (e.g., (A + B) - (C + D)), supporting flexible and complex contrast definitions.}

\item{widthlegend}{Width of the legend in \strong{volcano plots} (used only if
\code{method = "all"}) and violin score plots.}

\item{sig_threshold}{P-value cutoff shown as a \strong{guide line} in volcano
plots. Only applies when \code{method = "all"}.}

\item{cohen_threshold}{Effect size threshold shown as a \strong{guide line} in
volcano plots. Used only when \code{method = "all"}.}

\item{colorPalette}{Name of an RColorBrewer palette used to assign colors in
plots. Applies to all methods. Default is \code{"Set3"}. If \code{ColorValues} is
provided, it overrides this palette.
If \code{Variable} is \code{NULL} and \code{method != "all"} (i.e., for density plots),
a default color \code{"#ECBD78"} is used.
If \code{method = "all"} (i.e., for heatmaps and volcano plots), a default
diverging color scale is used: \code{c("#F9F4AE", "#B44141")}, unless \code{ColorValues}
is manually specified.}

\item{cor}{Correlation method for numeric variables. One of \code{"pearson"}
(default), \code{"spearman"}, or \code{"kendall"}. Only applies when the variable is
numeric and \code{method != "all"}.}
}
\value{
Depending on \code{method}:

If \code{method = "all"}, returns a list with \code{heatmap} and \code{volcano} ggplot objects.

If \code{method} is a single method, returns a single ggplot object (scatter or
violin plot depending on variable type).
}
\description{
Computes and visualizes gene signature scores using one or more methods,
returning plots such as scatter plots, violin plots, heatmaps, or volcano
plots depending on inputs.
}
\details{
Behavior based on \code{method}:

For \code{"all"}, the function requires \code{metadata} and \code{Variable}. It computes
scores using all available methods and returns a heatmap of Cohen's effect
sizes and a volcano plot showing effect size vs p-value across gene
signatures. Additional parameters include \code{mode} to define how contrasts
between groups are constructed, \code{sig_threshold} and \code{cohend_threshold} which
add guide dashed lines to the volcano plot (do not affect point coloring),
\code{widthlegend} controlling width of the volcano plot legend, and \code{pointSize}
controlling dot size for signature points in the volcano plot.
\code{ColorValues} can be a named list with \code{heatmap} (two-color gradient for
effect sizes) and \code{signatures} (named vector of colors for gene signatures
in the volcano plot).

For \code{"ssGSEA"}, \code{"logmedian"}, or \code{"ranking"}, the type of \code{Variable}
determines the plot. If categorical, violin plots with optional group
comparisons are produced. If numeric, scatter plots with correlation are
produced. If \code{Variable} is \code{NULL}, density plots for each signature across
all samples are produced. Additional arguments include \code{ColorVariable} and
\code{ColorValues} for coloring control, \code{colorPalette} (overridden by
\code{ColorValues} if present), \code{ConnectGroups} to link samples by ID for
categorical \code{Variable}, \code{cor} to specify correlation method for numeric
\code{Variable}, \code{pvalcalc} to enable group-wise p-value calculations for
categorical variables, \code{compute_cohen} to calculate effect sizes when
applicable, and \code{cond_cohend} to focus Cohen's d calculation on a specific
comparison.

Behavior based on \code{Variable} type:

If \code{Variable} is numeric, scatter plots are output (in single-method mode)
with computed correlation (\code{cor}). Parameters \code{compute_cohen},
\code{cond_cohend}, and \code{pvalcalc} are ignored. Color is uniform (default:
\code{"#5264B6"}) unless overridden via \code{ColorValues}. Cohen's f effect size
estimation (\code{compute_cohen = TRUE}) and significance if \code{pvalcalc} is \code{TRUE}.

If \code{Variable} is categorical, violin plots are output (in single-method mode)
supporting p-value comparisons (\code{pvalcalc = TRUE}), optional connection lines
(\code{ConnectGroups = TRUE}), and Cohen's effect size estimation
(\code{compute_cohen = TRUE}) with significance (\code{pvalcalc} is \code{TRUE}). If
\code{cond_cohend} is specified, computes Cohen's d for that comparison. If not
specified, computes Cohen's d if 2 groups or Cohen's f if more than 2 groups.
Colors are matched to factor levels using \code{ColorValues} or \code{colorPalette}.

If \code{Variable} is \code{NULL} and \code{method != "all"}, density plots of signature
scores are produced. A single fill color is used (default \code{"#ECBD78"} or
from \code{ColorValues}).
}
\examples{
# Simulate positive gene expression data (genes as rows, samples as columns)
set.seed(42)
expr <- as.data.frame(matrix(rexp(60, rate = 0.2), nrow = 6, ncol = 10))  # values > 0
rownames(expr) <- paste0("Gene", 1:6)
colnames(expr) <- paste0("Sample", 1:10)

# Simulate metadata for samples with categorical and numeric variables
metadata <- data.frame(
  sample = colnames(expr),
  Group = rep(c("A", "B"), each = 5),
  Age = seq(30, 75, length.out = 10)
)

# Define two simple gene sets
gene_sets <- list(
  Signature1 = c("Gene1", "Gene2", "Gene3"),
  Signature2 = c("Gene4", "Gene5", "Gene6")
)

# 1. Categorical variable: Violin plot (logmedian)
PlotScores(
  data = expr,
  metadata = metadata,
  gene_sets = gene_sets,
  method = "logmedian",
  Variable = "Group"
)

# 2. Numeric variable: Scatter plot (logmedian)
PlotScores(
  data = expr,
  metadata = metadata,
  gene_sets = gene_sets,
  method = "logmedian",
  Variable = "Age"
)

# 3. No variable: Density plot (logmedian)
PlotScores(
  data = expr,
  metadata = metadata,
  gene_sets = gene_sets,
  method = "logmedian"
)

# 4. All methods, categorical variable: Heatmap and volcano
# (Returns a list with $heatmap and $volcano elements)
all_plots <- PlotScores(
  data = expr,
  metadata = metadata,
  gene_sets = gene_sets,
  method = "all",
  Variable = "Group"
)
# Print the heatmap and volcano plot if desired
print(all_plots$heatmap)
print(all_plots$volcano)

}
