#' Writes maf summaries to output tab-delimited text files.
#' @details Writes MAF and related summaries to output files.
#'
#' @param maf an \code{\link{MAF}} object generated by \code{\link{read.maf}}
#' @param basename basename for output file to be written.
#' @param compress If `TRUE` files will be gz compressed. Default `FALSE`
#' @return None. Writes output as text files.
#' @seealso \code{\link{read.maf}}
#' @examples
#' laml.maf <- system.file("extdata", "tcga_laml.maf.gz", package = "maftools")
#' laml <- read.maf(maf = laml.maf)
#' write.mafSummary(maf = laml, basename = 'laml')
#'
#' @export

write.mafSummary = function(maf, basename = NULL, compress = FALSE){

  if(is.null(basename)){
    stop('Please provide a basename for output file.')
  }

  if(compress){
    gs = paste0(basename,'_geneSummary.txt.gz')
    ss = paste0(basename,'_sampleSummary.txt.gz')
    su = paste0(basename,'_summary.txt.gz')
    mf = paste0(basename,'_maftools.maf.gz')
    cd = paste0(basename,'_clinicalData.txt.gz')
  }else{
    gs = paste0(basename,'_geneSummary.txt')
    ss = paste0(basename,'_sampleSummary.txt')
    su = paste0(basename,'_summary.txt')
    mf = paste0(basename,'_maftools.maf')
    cd = paste0(basename,'_clinicalData.txt')
  }
  #write gene summary.
  write.table(x = getGeneSummary(maf), file = gs, sep='\t', quote = FALSE, row.names = FALSE)
  #write sample summary.
  write.table(x = getSampleSummary(maf), file = ss, sep='\t', quote = FALSE, row.names = FALSE)
  #write summary
  write.table(x = maf@summary,file = su, sep='\t', quote = FALSE, row.names = FALSE)
  #write main maf
  data.table::fwrite(x = data.table::rbindlist(list(maf@data, maf@maf.silent), use.names = TRUE, fill = TRUE),
                     file = mf, sep='\t',
                     quote = FALSE, row.names = FALSE)
  #write clinical
  data.table::fwrite(x = getClinicalData(maf),
                     file = cd, sep='\t',
                     quote = FALSE, row.names = FALSE)
}
