% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/vc_score_h_perm.R
\name{vc_score_h_perm}
\alias{vc_score_h_perm}
\title{Computes variance component score test statistics for homogeneous trajectory
and its permuted distribution}
\usage{
vc_score_h_perm(
  y,
  x,
  indiv,
  phi,
  w,
  Sigma_xi = diag(ncol(phi)),
  na_rm = FALSE,
  n_perm = 1000,
  progressbar = TRUE,
  parallel_comp = TRUE,
  nb_cores = parallel::detectCores(logical = FALSE) - 1
)
}
\arguments{
\item{y}{a numeric matrix of dim \code{g x n} containing the raw or
normalized RNA-seq counts for g genes from \code{n} samples.}

\item{x}{a numeric design matrix of dim \code{n x p} containing the \code{p}
covariates to be adjusted for.}

\item{indiv}{a vector of length \code{n} containing the information for
attributing each sample to one of the studied individuals. Coerced
to be a \code{factor}.}

\item{phi}{a numeric design matrix of size \code{n x K} containing the
\code{K} longitudinal variables to be tested (typically a vector of time
points or functions of time).}

\item{w}{a vector of length \code{n} containing the weights for the \code{n}
samples, corresponding to the inverse of the diagonal of the estimated
covariance matrix of y.}

\item{Sigma_xi}{a matrix of size \code{K x K} containing the covariance matrix
of the \code{K} random effects corresponding to \code{phi}.}

\item{na_rm}{logical: should missing values (including \code{NA} and
\code{NaN}) be omitted from the calculations? Default is \code{FALSE}.}

\item{n_perm}{the number of permutation to perform. Default is \code{1000}.}

\item{progressbar}{logical indicating wether a progressBar should be displayed
when computing permutations (only in interactive mode).}

\item{parallel_comp}{a logical flag indicating whether parallel computation
should be enabled. Only Linux and MacOS are supported, this is ignored on
Windows. Default is \code{TRUE}.}

\item{nb_cores}{an integer indicating the number of cores to be used when
\code{parallel_comp} is \code{TRUE}.
Default is \code{parallel::detectCores(logical=FALSE) - 1}.}
}
\value{
A list with the following elements:
\itemize{
  \item \code{score}: an approximation of the observed set score
  \item \code{scores_perm}: a vector containing the permuted set scores
  \item \code{gene_scores_unscaled}: approximation of the individual gene
  scores
  \item \code{gene_scores_unscaled_perm}: a list of approximation of the
  permuted individual gene scores
}
}
\description{
This function computes the variance component score test statistics
for homogeneous trajectories along with its permuted values for estimating
its distribution under the null hypothesis.
}
\examples{
set.seed(123)

##generate some fake data
########################
ng <- 100
nindiv <- 30
nt <- 5
nsample <- nindiv*nt
tim <- matrix(rep(1:nt), nindiv, ncol=1, nrow=nsample)
tim2 <- tim^2
sigma <- 5
b0 <- 10

#under the null:
beta1 <- rnorm(n=ng, 0, sd=0)
#under the (heterogen) alternative:
beta1 <- rnorm(n=ng, 0, sd=0.1)
#under the (homogen) alternative:
beta1 <- rnorm(n=ng, 0.06, sd=0)

y.tilde <- b0 + rnorm(ng, sd = sigma)
y <- t(matrix(rep(y.tilde, nsample), ncol=ng, nrow=nsample, byrow=TRUE) +
      matrix(rep(beta1, each=nsample), ncol=ng, nrow=nsample, byrow=FALSE) *
           matrix(rep(tim, ng), ncol=ng, nrow=nsample, byrow=FALSE) +
      #matrix(rep(beta1, each=nsample), ncol=ng, nrow=nsample, byrow=FALSE) *
      #     matrix(rep(tim2, ng), ncol=ng, nrow=nsample, byrow=FALSE) +
      matrix(rnorm(ng*nsample, sd = sigma), ncol=ng, nrow=nsample,
             byrow=FALSE)
      )
myindiv <- rep(1:nindiv, each=nt)
x <- cbind(1, myindiv/2==floor(myindiv/2))
myw <- matrix(rnorm(nsample*ng, sd=0.1), ncol=nsample, nrow=ng)

#run test
#We only use few permutations (10) to keep example running time low
#Otherwise one can use n_perm = 1000
score_homogen <- vc_score_h_perm(y, x, phi=tim, indiv=myindiv,
                                w=myw, Sigma_xi=cov(tim), n_perm = 10,
                                parallel_comp = FALSE)
score_homogen$score

score_heterogen <- vc_score_perm(y, x, phi=tim, indiv=myindiv,
                           w=myw, Sigma_xi=cov(tim), n_perm = 10,
                           parallel_comp = FALSE)
score_heterogen$score

scoreTest_homogen <- vc_test_asym(y, x, phi=tim, indiv=rep(1:nindiv, each=nt),
                                 w=matrix(1, ncol=ncol(y), nrow=nrow(y)),
                                 Sigma_xi=cov(tim), homogen_traj = TRUE)
scoreTest_homogen$set_pval
scoreTest_heterogen <- vc_test_asym(y, x, phi=tim, indiv=rep(1:nindiv,
                                                            each=nt),
                                   w=matrix(1, ncol=ncol(y), nrow=nrow(y)),
                                   Sigma_xi=cov(tim), homogen_traj = FALSE)
scoreTest_heterogen$set_pval

}
\keyword{internal}
