#' Class \code{\link{dagPeptides}}.
#' An S4 class to represent formatted, aligned peptides for dagLogo analysis.
#' 
#' @section Objects from the Class: 
#'  Objects can be created by calls of the form 
#'  
#'  \code{new("dagPeptides", data, peptides, upstreamOffset, downstreamOffset, type)}.
#' @slot data A data frame with column names: IDs, anchorAA, anchorPos, peptide and anchor.
#' @slot peptides A matrix of character, each element is a single-character symbol
#' for a amino acid.
#' @slot upstreamOffset An integer, the upstream offset
#' relative to the anchoring position.
#' @slot downstreamOffset An integer, the downstream offset relative to the
#' anchoring position.
#' @slot type A character vector of length 1. Available options :"UniProt", 
#' and "fasta" if the \code{\link{dagPeptides}} object is generated using 
#' the function \code{\link{formatSequence}}, or "entrezgene" and 
#' "uniprotswissprot" if generated by the function \code{\link{fetchSequence}}.
#' 
#' @name dagPeptides-class
#' @rdname dagPeptides-class
#' @aliases dagPeptides
#' @exportClass dagPeptides
#' @import methods
#' 
#' @author Jianhong Ou


setClass(
    "dagPeptides",
    representation(
        data = "data.frame",
        peptides = "matrix",
        upstreamOffset = "numeric",
        downstreamOffset = "numeric",
        type = "character"
    ),
    validity = function(object) {
        re <- TRUE
        if (object@upstreamOffset < 0 ||
            object@downstreamOffset < 0)
            re <-
                "upstreamOffset and downstreamOffset should be integers no less than 0."
        peptides <- as.character(object@peptides)
        peptides <-
            peptides[(!is.na(peptides)) & (peptides != "NA")]
        if (!all(1 == nchar(peptides)))
            re <-
            "peptides must be a matrix with one amino acid in each position."
        re
    }
)

#' Class \code{\link{Proteome}}.
#' 
#' An S4 class to represent a whole proteome for dagLogo analysis.
#' @section Objects from the Class:
#'  Objects can be created by calls of the form 
#'  
#'  \code{new("Proteome", proteome, type, species)}.
#'  
#' @slot proteome A data frame.
#' @slot type A character vector of length 1. Available options :"UniProt", and 
#' "fasta".
#' @slot species A character vector of length 1, such as a conventional Latin name for 
#' a species.
#'
#' @name Proteome-class
#' @rdname Proteome-class
#' @aliases Proteome
#' @exportClass Proteome
#' 
#' @import methods
#' 
#' @author Jianhong Ou

setClass(
    "Proteome",
    representation(
        proteome = "data.frame",
        type = "character",
        species = "character"
    ),
    validity = function(object) {
        re <- TRUE
        if (!object@type %in% c("fasta", "UniProt"))
            re <- "type must be 'fasta' or 'UniProt'"
        if (object@type == "UniProt" &&
            is.null(object@proteome$ID))
            re <-
                "When type equals to UniProt, ID column is required for Proteome."
        if (is.null(object@proteome$SEQUENCE))
            re <- "Proteome sequence is required."
        re
    }
)

#' Class \code{dagBackground}.
#' 
#' An S4 class to represent a background composed of a formatted, 
#' aligned peptides for dagLogo analysis.
#' 
#' @slot background A list of data frame, each of which represetns 
#' one subset of the background set. Within each n-by-1 dataframe is a the 
#' aligned peptides of same length.
#' @slot numSubsamples An integer. That is the length of the \code{background} list
#' @slot testType An character. The type of statistic testing for dagLogo analysis of 
#' differential usage of amino acids.
#' 
#' @name dagBackground-class
#' @rdname dagBackground-class
#' @aliases dagBackground
#' @exportClass dagBackground
#' @import methods
#' 
#' @author Jianhong Ou, Haibo Liu

setClass("dagBackground",
         representation(background = "list",
                        numSubsamples = "integer",
                        testType = "character"))


#' Class \code{testDAUresults}.
#' 
#' An S4 class to represent a DAU statistical test result from dagLogo analysis.
#' 
#' @slot group A character vector of length 1, the type of method for grouping 
#' amino acid.
#' @slot testType A character vector of length 1, the type of statistic testing.
#' The available options are "fisher" and "z-test".
#' @slot difference A numeric matrix consisting of differences of amino acid 
#' proportions between the test set and the background set of aligned, formatted 
#' peptides at each position.
#' @slot statistics A numeric matrix consisting of Z-scores or odds ratios for
#' Z-test and Fisher's exact test, respectively.
#' @slot pvalue A numeric matrix consisting of p-values.
#' @slot background A numeric matrix consisting of amino acid proportions in the 
#' background set of aligned, formatted peptides at each position.
#' @slot motif A numeric matrix consisting of amino acid proportions at each 
#' position for visualization by dagLogo. 
#' @slot upstreamOffset A positive integer, the upstream offset
#' relative to the anchoring position.
#' @slot downstreamOffset A positive integer, the upstream offset
#' relative to the anchoring position.
#' 
#' @name testDAUresults-class
#' @rdname testDAUresults-class
#' @aliases testDAUresults
#' @exportClass testDAUresults
#' @import methods
#' 
#' @author Jianhong Ou, Haibo Liu

setClass(
    "testDAUresults",
    representation(
        group = "character",
        testType = "character",
        difference = "matrix",
        statistics = "matrix",
        pvalue = "matrix",
        background = "matrix",
        motif = "matrix",
        upstreamOffset = "numeric",
        downstreamOffset = "numeric"
    ),
    validity = function(object) {
        re <- TRUE
        if (object@upstreamOffset < 0 || object@downstreamOffset < 0)
        {
            re <-
                "Both upstreamOffset and downstreamOffset should be positive integers."
        }

        if(!object@testType %in% c("fisher", "ztest"))
        {
            re <- 'The test type must be "fisher" or "ztest".'
        } else 
        {
            if (ncol(object@statistics) == 0 || ncol(object@difference) == 0 || 
                ncol(object@pvalue) == 0)
            {
                re <-
                    "Slots statistics, difference and pvalue could not be empty." 
            }
            if (any(dim(object@pvalue) != dim(object@difference)) || 
                any(dim(object@pvalue) != dim(object@statistics)))
            {
                re <-
                    "Dimensions of slots for statistics, difference and pvalue should be identical."
            }
        } 
        re
    }
)
