% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/findSpacerPairs.R
\name{findSpacerPairs}
\alias{findSpacerPairs}
\title{Find pairs of CRISPR gRNA spacers from a pair of genomic regions.}
\usage{
findSpacerPairs(
  x1,
  x2,
  sortWithinPair = TRUE,
  pamOrientation = c("all", "out", "in"),
  minCutLength = NULL,
  maxCutLength = NULL,
  crisprNuclease = NULL,
  bsgenome = NULL,
  canonical = TRUE,
  both_strands = TRUE,
  spacer_len = NULL,
  strict_overlap = TRUE,
  remove_ambiguities = TRUE
)
}
\arguments{
\item{x1}{Either a \linkS4class{GRanges}, a \linkS4class{DNAStringSet}, or a
\linkS4class{DNAString} object, or a character vector of genomic
sequences. This specifies the sequence space from which gRNAs in
position 1 of the pairs will be designed. Alternatively, a 
\linkS4class{GuideSet} object can be provided.}

\item{x2}{Either a \linkS4class{GRanges}, a \linkS4class{DNAStringSet}, or a
\linkS4class{DNAString} object, or a character vector of genomic
sequences. This specifies the sequence space from which gRNAs in
position 2 of the pairs will be designed. Alternatively, a 
\linkS4class{GuideSet} object can be provided.}

\item{sortWithinPair}{Should gRNAs be sorted by chr and position
within a pair? TRUE by default.}

\item{pamOrientation}{String specifying a constraint on the PAM orientation
of the pairs. Should be either "all" (default), "out" (for the so-called
PAM-out orientation) or "in" (for PAM-in orientation).}

\item{minCutLength}{Integer specifying the minimum cut length allowed
(distance between the two cuts) induced by the gRNA pair.
If NULL (default), the argument is ignored. Note that this parameter
is only applicable for pairs of gRNAs targeting the same chromosome.}

\item{maxCutLength}{Integer specifying the maximum cut length allowed
(distance between the two cuts) induced by the gRNA pair.
If NULL (default), the argument is ignored. Note that this parameter
is only applicable for pairs of gRNAs targeting the same chromosome.}

\item{crisprNuclease}{A \linkS4class{CrisprNuclease} object.}

\item{bsgenome}{A \linkS4class{BSgenome} object from which to extract
sequences if \code{x} is a \linkS4class{GRanges} object.}

\item{canonical}{Whether to return only guide sequences having canonical
PAM sequences. If TRUE (default), only PAM sequences with the highest
weights stored in the \code{crisprNuclease} object will be considered.}

\item{both_strands}{Whether to consider both strands in search for 
protospacer sequences. \code{TRUE} by default.}

\item{spacer_len}{Length of spacers to return, if different from the
default length specified by \code{crisprNuclease}.}

\item{strict_overlap}{Whether to only include gRNAs that cut in the input
range, as given by \code{cut_site} (\code{TRUE}) or to include all
gRNAs that share any overlap with the input range (\code{FALSE}).
\code{TRUE} by default. Ignored when \code{x} is not a
\linkS4class{GRanges} object.}

\item{remove_ambiguities}{Whether to remove spacer sequences that contain
ambiguous nucleotides (not explicily \code{A}, \code{C}, \code{G}, or
\code{T}). TRUE by default.}
}
\value{
A \linkS4class{PairedGuideSet} object.
}
\description{
Returns all possible, valid gRNA sequences for a given CRISPR
   nuclease from either a \linkS4class{GRanges} object or a set of
   sequence(s) contained in either a \linkS4class{DNAStringSet},
   \linkS4class{DNAString} or character vector of genomic sequences.
}
\details{
This function returns a \linkS4class{PairedGuideSet} object 
    that stores gRNA pairs targeting the two genomic regions provided as 
    input. The gRNAs in position 1 target the first genomic region,
    and the gRNAs in position 2 target the second genomic region.

    This function can be used for the following scenarios:
     
    1. Designing pairs of gRNAs targeting different genes, for instance
       for dual-promoter Cas9 systems, or polycystronic Cas12a constructs.
       This can also be used to target a given gene with multiple gRNAs
       for improved efficacy (for instance CRISPRa and CRISPRi)

    2. Designing pairs of gRNAs for double nicking systems such as Cas9 D10A.
    
    See vignette for more examples.
}
\examples{

library(GenomicRanges)
library(BSgenome.Hsapiens.UCSC.hg38)
library(crisprBase)
bsgenome <- BSgenome.Hsapiens.UCSC.hg38

# Region 1:
gr1 <- GRanges(c("chr12"),
               IRanges(start=22224014, end=22225007))

# Region 2:
gr2 <- GRanges(c("chr13"),
               IRanges(start=23224014, end=23225007))

# Pairs targeting the same region:
pairs <- findSpacerPairs(gr1, gr1, bsgenome=bsgenome)

# Pairs targeting two regions:
# The gRNA in position targets gr1
# and the gRNA in position 2 targets gr2
pairs <- findSpacerPairs(gr1, gr2, bsgenome=bsgenome)

}
\seealso{
\code{\link{findSpacers}} to find unpaired spacer sequences, and the 
    \code{PairedGuideSet} object documentation to understand the
    output of \code{findSpacerPairs}.
}
\author{
Jean-Philippe Fortin
}
