% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/plotClusters.R
\docType{methods}
\name{plotClusters}
\alias{plotClusters}
\alias{plotClusters,ClusterExperiment-method}
\alias{plotClusters,matrix-method}
\title{Visualize cluster assignments across multiple clusterings}
\usage{
\S4method{plotClusters}{ClusterExperiment}(
  object,
  whichClusters,
  existingColors = c("ignore", "all", "firstOnly"),
  resetNames = FALSE,
  resetColors = FALSE,
  resetOrderSamples = FALSE,
  colData = NULL,
  clusterLabels = NULL,
  ...
)

\S4method{plotClusters}{matrix}(
  object,
  orderSamples = NULL,
  colData = NULL,
  reuseColors = FALSE,
  matchToTop = FALSE,
  plot = TRUE,
  unassignedColor = "white",
  missingColor = "grey",
  minRequireColor = 0.3,
  startNewColors = FALSE,
  colPalette = massivePalette,
  input = c("clusters", "colors"),
  clusterLabels = colnames(object),
  add = FALSE,
  xCoord = NULL,
  ylim = NULL,
  tick = FALSE,
  ylab = "",
  xlab = "",
  axisLine = 0,
  box = FALSE,
  ...
)
}
\arguments{
\item{object}{A matrix of with each column corresponding to a clustering
and each row a sample or a \code{\link{ClusterExperiment}} object. If a
matrix, the function will plot the clusterings in order of this matrix, and
their order influences the plot greatly.}

\item{whichClusters}{argument that can be either numeric or character vector
indicating the clusterings to be used. See details of \code{\link{getClusterIndex}}.}

\item{existingColors}{how to make use of the exiting colors in the 
\code{ClusterExperiment} object. 'ignore' will ignore them and assign new 
colors. 'firstOnly' will use the existing colors of only the 1st 
clustering, and then align the remaining clusters and give new colors for
the remaining only. 'all' will use all of the existing colors.}

\item{resetNames}{logical. Whether to reset the names of the clusters in
\code{clusterLegend} to be the aligned integer-valued ids from
\code{plotClusters}.}

\item{resetColors}{logical. Whether to reset the colors in
\code{clusterLegend} in the \code{ClusterExperiment} returned to be the
colors from the \code{plotClusters}.}

\item{resetOrderSamples}{logical. Whether to replace the existing
\code{orderSamples} slot in the \code{ClusterExperiment} object with the
new order found.}

\item{colData}{If \code{clusters} is a matrix, \code{colData} gives a 
matrix of additional cluster/sample data on the samples to be plotted with 
the clusterings given in clusters. Values in \code{colData} will be 
added to the end (bottom) of the plot. NAs in the \code{colData} matrix
will trigger an error. If \code{clusters} is a \code{ClusterExperiment}
object, the input in \code{colData} refers to columns of the the
\code{colData} slot of the \code{ClusterExperiment} object to be plotted
with the clusters. In this case, \code{colData} can be TRUE (i.e. all
columns will be plotted), or an index or a character vector that references
a column or column name, respectively, of the \code{colData} slot of the
\code{ClusterExperiment} object. If there are NAs in the \code{colData}
columns, they will be encoded as 'unassigned' and receive the same color as
'unassigned' samples in the clustering.}

\item{clusterLabels}{names to go with the columns (clusterings) in matrix 
\code{colorMat}. If \code{colData} argument is not \code{NULL}, the 
\code{clusterLabels} argument must include names for the sample data too. 
If the user gives only names for the clusterings, the code will try to 
anticipate that and use the column names of the sample data, but this is
fragile. If set to \code{FALSE}, then no labels will be plotted.}

\item{...}{for \code{plotClusters} arguments passed either to the method
of \code{plotClusters} for matrices, or ultimately to \code{\link{plot}}
(if \code{add=FALSE}).}

\item{orderSamples}{A predefined order in which the samples will be plotted.
Otherwise the order will be found internally by aligning the clusters
(assuming \code{input="clusters"})}

\item{reuseColors}{Logical. Whether each row should consist of the same set
of colors. By default (FALSE) each cluster that the algorithm doesn't
identify to the previous rows clusters gets a new color.}

\item{matchToTop}{Logical as to whether all clusters should be aligned to the
first row. By default (FALSE) each cluster is aligned to the ordered
clusters of the row above it.}

\item{plot}{Logical as to whether a plot should be produced.}

\item{unassignedColor}{If ``-1'' in \code{clusters}, will be given this color
(meant for samples not assigned to cluster).}

\item{missingColor}{If ``-2'' in clusters, will be given this color (meant
for samples that were missing from the clustering, mainly when comparing
clusterings run on different sets of samples)}

\item{minRequireColor}{In aligning colors between rows of clusters, require
this percent overlap.}

\item{startNewColors}{logical, indicating whether in aligning colors between
rows of clusters, should the colors restart at beginning of colPalette as
long as colors are not in immediately proceeding row (the colors at the end 
of \code{massivePalette} are all of \code{\link{colors}()} and many will be
indistinguishable, so this option can be useful if you have a large cluster
matrix).}

\item{colPalette}{a vector of colors used for the different clusters. Must be
as long as the maximum number of clusters found in any single
clustering/column given in \code{clusters} or will otherwise return an
error.}

\item{input}{indicate whether the input matrix is matrix of integer assigned
clusters, or contains the colors. If \code{input="colors"}, then the object
\code{clusters} is a matrix of colors and there is no alignment (this
option allows the user to manually adjust the colors and replot, for
example).}

\item{add}{whether to add to existing plot.}

\item{xCoord}{values on x-axis at which to plot the rows (samples).}

\item{ylim}{vector of limits of y-axis.}

\item{tick}{logical, whether to draw ticks on x-axis for each sample.}

\item{ylab}{character string for the label of y-axis.}

\item{xlab}{character string for the label of x-axis.}

\item{axisLine}{the number of lines in the axis labels on y-axis should be
(passed to \code{line = ...} in the axis call).}

\item{box}{logical, whether to draw box around the plot.}
}
\value{
If \code{clusters} is a \code{\link{ClusterExperiment}} Object, then
  \code{plotClusters} returns an updated \code{ClusterExperiment} object,
  where the \code{clusterLegend} and/or \code{orderSamples} slots have been
  updated (depending on the arguments).

If \code{clusters} is a matrix, \code{plotClusters} returns
  (invisibly) the orders and other things that go into making the matrix.
  Specifically, a list with the following elements.
\itemize{

\item{\code{orderSamples}}{ a vector of length equal to 
\code{nrows(clusters)} giving the order of the samples (rows) to 
 use to get the original clusters matrix
into the order made by \code{plotClusters}.}

\item{\code{colors}}{ matrix of color assignments for each element of
original clusters matrix. Matrix is in the same order as original clusters
matrix. The matrix \code{colors[orderSamples,]} is the matrix that can be 
given back to \code{plotClusters} to recreate the plot (see examples).}

\item{\code{alignedClusterIds}}{ a matrix of integer valued cluster
assignments that match the colors. This is useful if you want to have cluster
identification numbers that are better aligned than that given in the
original clusters. Again, the rows/samples are in same order as original 
input matrix.}

\item{\code{clusterLegend}}{ list of length equal to the number of columns of
input matrix. The elements of the list are matrices, each with three columns
named "Original","Aligned", and "Color" giving, respectively, the
correspondence between the original cluster ids in \code{clusters}, the
aligned cluster ids in \code{aligned}, and the color.}

\item{\code{origClusters}}{The original matrix of clusters given to
\code{plotClusters}}
}
}
\description{
Align multiple clusterings of the same set of samples and provide a
color-coded plot of their shared cluster assignments
}
\details{
All arguments of the matrix version can be passed to the
  \code{ClusterExperiment} version. As noted above, however, some arguments
  have different interpretations.

If \code{whichClusters = "workflow"}, then the workflow clusterings
  will be plotted in the following order: final, mergeClusters, makeConsensus,
  clusterMany.
}
\examples{
\dontrun{
#clustering using pam: try using different dimensions of pca and different k
data(simData)

cl <- clusterMany(simData, nReducedDims=c(5, 10, 50), reduceMethod="PCA",
clusterFunction="pam", ks=2:4, findBestK=c(TRUE,FALSE),
removeSil=c(TRUE,FALSE), makeMissingDiss=TRUE)

clusterLabels(cl)

#make names shorter for better plotting
x <- clusterLabels(cl)
x <- gsub("TRUE", "T", x)
x <- gsub("FALSE", "F", x)
x <- gsub("k=NA,", "", x)
x <- gsub("Features", "", x)
clusterLabels(cl) <- x

par(mar=c(2,10,1,1))
#this will make the choices of plotClusters
cl <- plotClusters(cl, axisLine=-1, resetOrderSamples=TRUE, resetColors=TRUE)

#see the new cluster colors
clusterLegend(cl)[1:2]

#We can also change the order of the clusterings. Notice how this
#dramatically changes the plot!
clOrder <- c(3:6, 1:2, 7:ncol(clusterMatrix(cl)))
cl <- plotClusters(cl, whichClusters=clOrder, resetColors=TRUE,
resetOrder=TRUE, axisLine=-2)

#We can manually switch the red ("#E31A1C") and green ("#33A02C") in the
#first cluster:

#see what the default colors are and their names
showPalette(wh=1:5)

#change "#E31A1C" to "#33A02C"
newColorMat <- clusterLegend(cl)[[clOrder[1]]]
newColorMat[2:3, "color"] <- c("#33A02C", "#E31A1C")
clusterLegend(cl)[[clOrder[1]]]<-newColorMat

#replot by setting 'input="colors"'
par(mfrow=c(1,2))
plotClusters(cl, whichClusters=clOrder, orderSamples=orderSamples(cl),
existingColors="all")
plotClusters(cl, whichClusters=clOrder, resetColors=TRUE, resetOrder=TRUE,
axisLine=-2)
par(mfrow=c(1,1))

#set some of clusterings arbitrarily to "-1", meaning not clustered (white),
#and "-2" (another possible designation getting gray, usually for samples not
#included in original clustering)
clMatNew <- apply(clusterMatrix(cl), 2, function(x) {
wh <- sample(1:nSamples(cl), size=10)
x[wh]<- -1
wh <- sample(1:nSamples(cl), size=10)
x[wh]<- -2
return(x)
})

#make a new object
cl2 <- ClusterExperiment(assay(cl), clMatNew,
transformation=transformation(cl))
plotClusters(cl2)
}
}
\references{
Wilkerson, D. M, Hayes and Neil D (2010). "ConsensusClusterPlus: a class discovery tool with confidence assessments and item tracking." Bioinformatics, 26(12), pp. 1572-1573.
}
\seealso{
The \code{\link[ConsensusClusterPlus]{ConsensusClusterPlus}} package.
}
\author{
Elizabeth Purdom and Marla Johnson (based on the tracking plot in
  \link[ConsensusClusterPlus]{ConsensusClusterPlus} by Matt Wilkerson and Peter
  Waltman).
}
