## ----library, include=FALSE---------------------------------------------------
library(rtracklayer)
library(R.utils)
library(liftOver)

## ----install, eval=FALSE------------------------------------------------------
# if (!"BiocManager" %in% rownames(installed.packages())) {
#     install.packages("BiocManager")
# }
# BiocManager::install("bedbaser")

## ----bedbase------------------------------------------------------------------
library(bedbaser)

bedbase <- BEDbase(tempdir())

## ----bb_list_beds-------------------------------------------------------------
bb_list_beds(bedbase)

bb_list_bedsets(bedbase)

## ----bb_metadata--------------------------------------------------------------
ex_bed <- bb_example(bedbase, "bed")
md <- bb_metadata(bedbase, ex_bed$id)
head(md)

## ----bb_beds_in_bedset--------------------------------------------------------
bb_beds_in_bedset(bedbase, "excluderanges")

## ----bb_search----------------------------------------------------------------
bb_bed_text_search(bedbase, "cancer", limit = 10)

## ----bb_to_granges------------------------------------------------------------
ex_bed <- bb_example(bedbase, "bed")
# Allow bedbaser to assign column names and types
bb_to_granges(bedbase, ex_bed$id, quietly = FALSE)

## ----bb_to_granges_manual, eval=FALSE-----------------------------------------
# # Manually assign column name and type using `extra_cols`
# bb_to_granges(bedbase, ex_bed$id, extra_cols = c("column_name" = "character"))

## ----bb_to_granges_narrowpeak, message=FALSE----------------------------------
bed_id <- "bbad85f21962bb8d972444f7f9a3a932"
md <- bb_metadata(bedbase, bed_id)
head(md)
bb_to_granges(bedbase, bed_id)

## ----bb_to_grangeslist, message=FALSE-----------------------------------------
bedset_id <- "lola_hg38_ucsc_features"
bb_to_grangeslist(bedbase, bedset_id)

## ----bb_save------------------------------------------------------------------
bb_save(bedbase, ex_bed$id, tempdir())

## ----operations---------------------------------------------------------------
show(bedbase)

## ----service_class_example----------------------------------------------------
library(httr)
show(bedbase$get_bed_stats_v1_bed__bed_id__metadata_stats_get)
id <- "bbad85f21962bb8d972444f7f9a3a932"
rsp <- bedbase$get_bed_stats_v1_bed__bed_id__metadata_stats_get(id)
content(rsp)

## ----liftOver, eval=FALSE-----------------------------------------------------
# if (!"BiocManager" %in% rownames(installed.packages())) {
#     install.packages("BiocManager")
# }
# BiocManager::install(c("liftOver", "rtracklayer"))
# 
# library(liftOver)
# library(rtracklayer)

## ----convertCoordinates_createGRangesObject, message=FALSE--------------------
id <- "f2a5b06011706376560514c3f39648ea"
bedbase <- BEDbase()
gro <- bb_to_granges(bedbase, id)
gro

## ----convertCoordinates_getTheChain, message=FALSE----------------------------
chain_url <- paste0(
    "https://hgdownload.cse.ucsc.edu/goldenPath/mm10/liftOver/",
    "mm10ToMm39.over.chain.gz"
)
tmpdir <- tempdir()
gz <- file.path(tmpdir, "mm10ToMm39.over.chain.gz")
download.file(chain_url, gz)
gunzip(gz, remove = FALSE)

## ----convertCoordinates_convert, message=FALSE--------------------------------
ch <- import.chain(file.path(tmpdir, "mm10ToMm39.over.chain"))
seqlevelsStyle(gro) <- "UCSC"
gro39 <- liftOver(gro, ch)
gro39 <- unlist(gro39)
genome(gro39) <- "mm39"
gro39

## ----sessionInfo--------------------------------------------------------------
sessionInfo()

