% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/aggTSE.R
\name{aggTSE}
\alias{aggTSE}
\title{Perform data aggregations based on the available tree structures}
\usage{
aggTSE(
  x,
  rowLevel = NULL,
  rowBlock = NULL,
  colLevel = NULL,
  colBlock = NULL,
  rowFun = sum,
  colFun = sum,
  whichRowTree = 1,
  whichColTree = 1,
  whichAssay = NULL,
  message = FALSE,
  rowDataCols,
  colDataCols,
  rowFirst = TRUE,
  BPPARAM = NULL
)
}
\arguments{
\item{x}{A \code{TreeSummarizedExperiment} object.}

\item{rowLevel}{A numeric (node numbers) or character (node labels) vector.
It provides the level on the tree that data is aggregated to. The
aggregation is on the row dimension. The default is \code{rowLevel = NULL},
and no aggregation is performed.}

\item{rowBlock}{A column name in the \code{rowData} to separate the
aggregation.}

\item{colLevel}{A numeric (node numbers) or character (node labels) vector.
It provides the level on the tree that data is aggregated to. The
aggregation is on the column dimension. The default is \code{colLevel =
NULL}, and no aggregation is performed.}

\item{colBlock}{A column name in the \code{colData} to separate the
aggregation.}

\item{rowFun}{A function to be applied on the row aggregation. It's similar
to the \code{FUN} in \code{\link[base]{apply}}.}

\item{colFun}{A function to be applied on the col aggregation. It's similar
to the \code{FUN} in \code{\link[base]{apply}}.}

\item{whichRowTree}{A integer scalar or string indicating which row tree is
used in the aggregation. The first row tree is used as default.}

\item{whichColTree}{A integer scalar or string indicating which row tree is
used in the aggregation. The first row tree is used as default.}

\item{whichAssay}{A integer scalar or string indicating which assay of
\code{x} to use in the aggregation. If NULL, all assay tables are used in
aggregation.}

\item{message}{A logical value. The default is TRUE. If TRUE, it will print
out the running process.}

\item{rowDataCols}{The rowData columns to include.}

\item{colDataCols}{The colData columns to include.}

\item{rowFirst}{TRUE or FALSE. If the aggregation is in both dims., it is
performed firstly on the row dim for \code{rowFirst = TRUE} or on the
column dim for \code{rowFirst = FALSE}.}

\item{BPPARAM}{Default is \code{NULL} and the computation isn't run in
parallel. To run computation parallelly, an optional
\code{\link[BiocParallel:BiocParallelParam-class]{BiocParallelParam}}
instance determining the parallel back-end to be used during evaluation, or
a list of BiocParallelParam instances, to be applied in sequence for nested
calls to \strong{BiocParallel} functions.}
}
\value{
A \code{\link{TreeSummarizedExperiment}} object
}
\description{
\code{aggTSE} aggregates values on the leaf nodes of a tree to a specific
arbitrary level of the tree. The level is specified via the nodes of the
tree. Users could decide on which dimension (row or column) and how should
the aggregation be performed.
}
\examples{

# assays data
set.seed(1)
toyTable <- matrix(rnbinom(20, size = 1, mu = 10), nrow = 5)
colnames(toyTable) <- paste(rep(LETTERS[1:2], each = 2),
                            rep(1:2, 2), sep = "_")
rownames(toyTable) <- paste("entity", seq_len(5), sep = "")

toyTable

# the column data
colInf <- DataFrame(gg = c(1, 2, 3, 3),
                    group = rep(LETTERS[1:2], each = 2),
                    row.names = colnames(toyTable))
colInf

# the toy tree
library(ape)
set.seed(4)
treeC <- rtree(4)
treeC$node.label <- c("All", "GroupA", "GroupB")

library(ggtree)
ggtree(treeC, size = 2) +
   geom_text2(aes(label = node), color = "darkblue",
           hjust = -0.5, vjust = 0.7, size = 6) +
    geom_text2(aes(label = label), color = "darkorange",
               hjust = -0.1, vjust = -0.7, size = 6)


tse <- TreeSummarizedExperiment(assays = list(toyTable),
                                colData = colInf,
                                colTree = treeC,
                                colNodeLab = treeC$tip.label,
                                metadata = list(test = 1:4))

aggCol <- aggTSE(x = tse, colLevel = c("GroupA", "GroupB"),
                 colFun = sum)

assays(aggCol)[[1]]

}
\author{
Ruizhu HUANG
}
