#' Updating Time Index correction
#'
#' This function can be used to correct or adjust the detected retention time
#' index (RI) markers or their location to specific retention times. This
#' function adds on [fixRI()] as it also corrects the RI of the CDF files
#'
#' Sometimes the retention time of the RI markers are not detected correctly,
#' either because there was a problem with the standard, or the time limits of
#' the [tsRim-class] object were not set correctly, or simply because the markers
#' are not injected with the samples.
#'
#' In any case, the retention time correction can be fixed by calling this
#' function. This function works almost exactly like [fixRI()], in fact, it
#' is called internally, and allows correction of RIfiles and CDFfiles at the
#' same time. Check also the documentation of [fixRI()] for extra details.
#'
#' The parameters are the [tsSample-class] and the [tsRim-class] object, with
#' optionally a `RImatrix` to force the location of the markers. The parameter
#' `quiet` can be unset to show what samples are corrected.
#'
#' If only a subset of samples require correction, then they can be chosen by
#' subsetting the object `sample`.
#'
#' Neededless to say, this function expect that the CDF files exists and are
#' in the TargetSearch format. If this is not the case, then use the function
#' [fixRI()], as this function deals only with RI files.
#'
#' @note
#' It is required that all the sample names of `samples` are contained in the
#' colnames of `RImatrix`, but the reverse is not necessary. The number of
#' columns of the output matrix will match the number of samples. Extra columns
#' in `RImatrix` will be ignored and not returned.
#'
#' @param samples A [tsSample-class] object created by [ImportSamples].
#' @param rimLimits A [tsRim-class] object. See [ImportFameSettings].
#' @param RImatrix An optional matrix. It represents a retention time matrix of
#'        the detected retention time markers that was obtained after running
#'        [RIcorrect]
#' @param quiet Logical. Do not print a list of converted files.
#' @return The retention index matrix. If `RImatrix` is not `NULL`, then the
#'        output is the same matrix.
#' @seealso [fixRI()], [RIcorrect()], [ImportSamples()], [ImportFameSettings()]
#' @author Alvaro Cuadros-Inostroza
#' @md
#' @examples
#' require(TargetSearchData)
#' # import refLibrary, rimLimits and sampleDescription.
#' data(TSExample)
#' CDFpath(sampleDescription) <-
#'     file.path(find.package("TargetSearchData"), "gc-ms-data")
#'
#' # convert a subset of files to netCDF4
#' smp <- ncdf4Convert(sampleDescription[1:6], path=".")
#'
#' # make a copy of the RI markers object
#' fames <- rimLimits
#'
#' # mess up the limits of marker 3 (real value is 369 seconds app.)
#' rimLimits(fames)[3,] <- c(375, 400)
#'
#' # run RIcorrect (skip CDF-4 conversion)
#' RImat <- RIcorrect(smp, fames, Window = 15, IntThreshold = 200)
#'
#' # fix the limits of marker 3
#' rimLimits(fames)[3,] <- c(360, 380)
#'
#' # update RI files and CDF files
#' RImat <- updateRI(smp, fames)
#'
#' # Pass a RI matrix for manual adjustment
#' RImat[, 3] <- c(252, 311, 369)
#' RImat <- updateRI(smp, fames, RImat)
#'
#' # To select specific samples, simply use sample subsetting
#' # Note, RImat2 has only one column in this case.
#' ( RImat2 <- updateRI(smp[3], fames, RImat) )
#'
`updateRI` <-
function(samples, rimLimits, RImatrix=NULL, quiet=TRUE)
{
    assert_that(is.flag(quiet))
    assert_that(inherits(samples, 'tsSample'))
    assert_that(inherits(rimLimits, 'tsRim'))

    # check for nc4 files
    if(!all(sapply(CDFfiles(samples), .is_ts_ncdf4)))
        stop(paste("All CDF files must be in TS format. Consider using the",
                  "function `fixRI` if the CDF files are not available, or",
                  "call the function `ncdf4Convert`"))

    RImatrix <- if(is.null(RImatrix)) riMatrix(samples, rimLimits) else
        .validateRImatrix(samples, rimLimits, RImatrix)

    # update CDF files
    temp <- mapply(function(f, k)
                   ncdf4_update_ri(f, RImatrix[, k], rimStandard(rimLimits)),
                   CDFfiles(samples), seq(ncol(RImatrix)))
    # update RI files
    temp <- fixRI(samples, rimLimits, RImatrix, quiet=quiet)
    invisible(RImatrix)
}

# vim: set ts=4 sw=4 et:
