% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/reportEdges.R
\name{reportEdges}
\alias{reportEdges}
\alias{reportEdges,ANY-method}
\alias{reportEdges,SummarizedExperiment-method}
\alias{reportEdges,SingleCellExperiment-method}
\title{Report MST edge coordinates}
\usage{
reportEdges(x, ...)

\S4method{reportEdges}{ANY}(x, mst, clusters, combined = TRUE, columns = NULL)

\S4method{reportEdges}{SummarizedExperiment}(x, ..., assay.type = "logcounts")

\S4method{reportEdges}{SingleCellExperiment}(
  x,
  clusters = colLabels(x, onAbsence = "error"),
  ...,
  use.dimred = NULL
)
}
\arguments{
\item{x}{A numeric matrix of coordinates where each row represents a cell/sample and each column represents a dimension
(usually a PC or another low-dimensional embedding, but features or genes can also be used).

Alternatively, a \linkS4class{SummarizedExperiment} or \linkS4class{SingleCellExperiment} object
containing such a matrix in its \code{\link{assays}}, as specified by \code{assay.type}.
This will be transposed prior to use.

Alternatively, for \linkS4class{SingleCellExperiment}s, this matrix may be extracted from its \code{\link{reducedDims}},
based on the \code{use.dimred} specification.
In this case, no transposition is performed.

Alternatively, if \code{clusters=NULL}, a numeric matrix of coordinates for cluster centroids,
where each row represents a cluster and each column represents a dimension 
Each row should be named with the cluster name.
This mode can also be used with assays/matrices extracted from SummarizedExperiments and SingleCellExperiments.}

\item{...}{For the generic, further arguments to pass to the specific methods.

For the SummarizedExperiment method, further arguments to pass to the ANY method.

For the SingleCellExperiment method, further arguments to pass to the SummarizedExperiment method
(if \code{use.dimred} is specified) or the ANY method (otherwise).}

\item{mst}{A \link{graph} object containing a MST, typically the output of \code{\link{createClusterMST}}.
This need not be constructed from the same coordinates as those in \code{x}.}

\item{clusters}{A factor-like object of the same length as \code{nrow(x)},
specifying the cluster identity for each cell in \code{x}.
If \code{NULL}, \code{x} is assumed to already contain coordinates for the cluster centroids.

Alternatively, a matrix with number of rows equal to \code{nrow(x)}, 
containing soft assignment weights for each cluster (column).
All weights should be positive and sum to 1 for each row.}

\item{combined}{Logical scalar indicating whether a single data.frame of edge coordinates should be returned.}

\item{columns}{A character, logical or integer vector specifying the columns of \code{x} to use.
If \code{NULL}, all provided columns are used by default.}

\item{assay.type}{An integer or string specifying the assay to use from a SummarizedExperiment \code{x}.}

\item{use.dimred}{An integer or string specifying the reduced dimensions to use from a SingleCellExperiment \code{x}.}
}
\value{
A data.frame containing the start and end coordinates of segments representing all the edges in \code{mst}.
If \code{combined=FALSE}, a list of two data.frames is returned where corresponding rows represent the start and end coordinates of the same edge.
}
\description{
Provides the coordinates of the start and end of every edge in the MST, 
possibly on a different coordinate space from that used to construct the MST.
This is mostly useful for plotting purposes in \code{\link{segments}} or the equivalent \pkg{ggplot2} functionality.
}
\details{
It is entirely possibly to supply, say, t-SNE coordinates in \code{x} along with a MST constructed from the PCA coordinates.
This allows us to visualize the edges of the MST on other low-dimensional embeddings.
The coordinates in \code{x} can be per-cell or, if \code{clusters=NULL}, they are assumed to already be per-cluster means.
\code{x} may also be \code{NULL}, in which case the center coordinates are obtained 
from the \code{coordinates} vertex attribute of \code{mst}.
}
\examples{
# Mocking up a Y-shaped trajectory.
centers <- rbind(c(0,0), c(0, -1), c(1, 1), c(-1, 1))
rownames(centers) <- seq_len(nrow(centers))
clusters <- sample(nrow(centers), 1000, replace=TRUE)
cells <- centers[clusters,]
cells <- cells + rnorm(length(cells), sd=0.5)

# Creating the MST:
mst <- createClusterMST(cells, clusters)

# Plotting the MST on top of existing visualizations:
edges <- reportEdges(x=NULL, mst, combined=FALSE)
plot(cells[,1], cells[,2], col=clusters)
segments(edges$start$dim1, edges$start$dim2, edges$end$dim1, 
     edges$end$dim2, lwd=5)

# Use with coordinates other than those used to make the MST:
shifted.cells <- cells + 10

shift.edges <- reportEdges(shifted.cells, mst, 
    clusters=clusters, combined=FALSE)
plot(shifted.cells[,1], shifted.cells[,2], col=clusters)
segments(shift.edges$start$dim1, shift.edges$start$dim2, 
    shift.edges$end$dim1, shift.edges$end$dim2, lwd=5)

# Also works for ggplot2:
df <- data.frame(shifted.cells, cluster=factor(clusters))
shift.edges2 <- reportEdges(shifted.cells, mst, clusters=clusters)

library(ggplot2)
ggplot(df) +
   geom_point(aes(x=X1, y=X2, color=cluster)) + 
   geom_line(data=shift.edges2, mapping=aes(x=dim1, y=dim2, group=edge))

}
\references{
Ji Z and Ji H (2016).
TSCAN: Pseudo-time reconstruction and evaluation in single-cell RNA-seq analysis.
\emph{Nucleic Acids Res.} 44, e117
}
\seealso{
\code{\link{createClusterMST}}, to generate \code{mst}.

\code{\link{quickPseudotime}}, a wrapper to quickly perform these calculations.
}
\author{
Aaron Lun
}
