###########################################################################
#' output a spectrum library into a PeakView format file
#' @param dat A data frame of a spectrum library
#' @param filename A character string for the name of the output.
#' @param nodup A logic value, indicating if remove duplicated 
#' sprectrum (default)
#' @param format A character string representing the output format. One of
#' "peakview" (default) and "openswath".
#' @return a file with the specified file name (lib.txt as default) will be
#' saved under the current working directory
#' @examples 
#' file <- paste(system.file("files",package="SwathXtend"),"Lib1.txt",sep="/")
#' dat <- readLibFile(file)
#' outputLib(dat)
############################################################################ 

outputLib <- function(dat, filename = "NewLib.txt", format = c("peakview", "openswath"), nodup = TRUE)
{
  if(nodup) dat <- dat[!duplicated(dat),]
  
  dat <- libraryFormat(dat)
  
  format <- match.arg(format)
  if(format == "peakview"){
    dat.res <- dat
  } else if (format == "openswath"){
    PrecursorMz  <- as.numeric(as.character(dat$Q1))
    ProductMz  <- as.numeric(as.character(dat$Q3))
    Tr_recalibrated  <- as.numeric(as.character(dat$RT_detected))
    ProteinName  <- dat$protein_name
    GroupLabel  <- as.factor(dat$isotype)
    LibraryIntensity  <- as.numeric(as.character(dat$relative_intensity))
    PeptideSequence  <- dat$stripped_sequence
    FullUniModPeptideName  <- dat$modification_sequence
    UniprotID  <- dat$uniprot_id
    decoy  <- as.factor(dat$decoy)
    PrecursorCharge  <- as.numeric(as.character(dat$prec_z))
    FragmentType  <- dat$frg_type
    FragmentCharge  <- as.numeric(as.character(dat$frg_z))
    FragmentSeriesNumber  <- as.numeric(as.character(dat$frg_nr))

    
    dat.res <- data.frame(PrecursorMz,ProductMz,Tr_recalibrated,
                          ProteinName,GroupLabel,LibraryIntensity,
                          PeptideSequence,FullUniModPeptideName,
                          UniprotID,decoy,PrecursorCharge,
                          FragmentType,FragmentCharge,FragmentSeriesNumber)
  
 
  } else {
    stop("Wrong output format!")
  }
  
  write.table(dat.res, file = filename,sep="\t",quote=FALSE,row.names=FALSE)

  
}