% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/decontamination.R
\name{spotclean}
\alias{spotclean}
\alias{spotclean.SummarizedExperiment}
\alias{spotclean.SpatialExperiment}
\title{Decontaminate spot swapping effect in spatial transcriptomics data}
\usage{
spotclean(slide_obj, ...)

\method{spotclean}{SummarizedExperiment}(
  slide_obj,
  gene_keep = NULL,
  maxit = 30,
  tol = 1,
  candidate_radius = 5 * seq_len(6),
  kernel = "gaussian",
  verbose = TRUE,
  ...
)

\method{spotclean}{SpatialExperiment}(
  slide_obj,
  gene_keep = NULL,
  gene_cutoff = 0.1,
  maxit = 30,
  tol = 1,
  candidate_radius = 5 * seq_len(6),
  kernel = "gaussian",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{slide_obj}{A slide object created or inherited from
\code{createSlide()}, or a \code{SpatialExperiment} object created from
\code{SpatialExperiment::read10xVisium()}.}

\item{...}{Arguments passed to other methods}

\item{gene_keep}{(vector of chr) Gene names to keep for decontamination.
We recommend not decontaminating lowly expressed and lowly variable genes
in order to save computation time. Even if user include them, their
decontaminated expressions will not change too much from raw expressions.
When setting to \code{NULL}, \code{keepHighGene()} will be
automatically called to filter out lowly expressed and lowly variable genes
before decontamination.
Default: \code{NULL}.}

\item{maxit}{(int) Maximum iteration for EM parameter updates. Default: 30.}

\item{tol}{(num) Tolerance to define convergence in EM parameter updates.
When the element-wise maximum difference between current and updated
parameter matrix is less than \code{tol}, parameters are considered
converged. Default: 1}

\item{candidate_radius}{(vector of num) Candidate contamination radius.
A series of radius to try when estimating contamination parameters.
Default: {c(5, 10, 15, 20, 25, 30)}}

\item{kernel}{(chr): name of kernel to use to model local contamination.
Supports "gaussian", "linear", "laplace", "cauchy". Default: "gaussian".}

\item{verbose}{(logical) Whether print progress information.
Default: \code{TRUE}}

\item{gene_cutoff}{(num) Filter out genes with average expressions
among tissue spots below or equal to this cutoff.
Only applies to \code{SpatialExperiment} object.
Default: 0.1.}
}
\value{
For slide object created from \code{createSlide()}, returns a
slide object where the decontaminated expression matrix is in the
"decont" assay slot and the contamination statistics are in
metadata slots. Contamination statistics include ambient RNA contamination
(ARC) score, bleeding rate, distal rate, contamination radius,
contamination kernel weight matrix, log-likelihood value in each iteration,
estimated proportion of contamination in each tissue spot in observed data.
Since decontaminated and raw data have different number of columns, they can
not be stored in a single object.

For \code{SpatialExperiment} object created from
\code{SpatialExperiment::read10xVisium()}, returns a
\code{SpatialExperiment} object where the decontaminated expression matrix
is in the "decont" assay slot and the contamination statistics are in
metadata slots. Raw expression matrix is also stored in the "counts" assay
slot. Genes are filtered based on \code{gene_cutoff}.
}
\description{
This is the main function implementing the SpotClean method
for decontaminating spot swapping effect in spatial transcriptomics data.
}
\details{
Briefly, the contamination level for the slide is estimated based on
the total counts of all spots. UMI counts travelling around the slide are
assumed to follow Poisson distributions and modeled by a mixture of
Gaussian (proximal) and uniform (distal) kernels. The underlying
uncontaminated gene expressions are estimated by EM algorithm to
maximize the data likelihood. Detailed derivation can be found in our
manuscript.
}
\examples{

data(mbrain_raw)
spatial_dir <- system.file(file.path("extdata",
                                     "V1_Adult_Mouse_Brain_spatial"),
                           package = "SpotClean")
mbrain_slide_info <- read10xSlide(tissue_csv_file=file.path(spatial_dir,
                                       "tissue_positions_list.csv"),
             tissue_img_file = file.path(spatial_dir,
                                       "tissue_lowres_image.png"),
             scale_factor_file = file.path(spatial_dir,
                                       "scalefactors_json.json"))
mbrain_obj <- createSlide(mbrain_raw,
                          mbrain_slide_info)

mbrain_decont_obj <- spotclean(mbrain_obj, tol=10, candidate_radius=20)
mbrain_decont_obj
}
