### ============================================================================
### [02_set_stat_model] accessors and methods for RflomicsMAE and RflomicsSE classes
### ----------------------------------------------------------------------------
# D. Charif,
# C. Payasant-Le-roux, 
# N. Bessoltane

#' @importFrom magrittr "%>%" 
NULL

# ---- generateModelFormulae ----
#' @title Statistical model formulae
#' @name generateModelFormulae
#' @aliases generateModelFormulae,RflomicsMAE-method
#' @rdname generateModelFormulae
#' @description
#' These methods allow the user to select and set the formula of 
#' the statistical method.
#' \itemize{
#'    \item generateModelFormulae():
#'  From a vector of character giving the name of the factors of an omics
#'  experiment, and their type of effect: biological or batch, it returns all 
#'  models formulae that can be formulated in association with this factors. 
#'  Batch effect factors do not appear in interaction terms with biological 
#'  factor. Model formulae stop in second order interaction.}
#' @param object an object of class \link{RflomicsSE} or 
#' class \link{RflomicsMAE-class}
#' @return a named list of object of class formula
#' @exportMethod generateModelFormulae
#' @example inst/examples/statSetting.R
setMethod(f          = "generateModelFormulae",
          signature  = "RflomicsMAE",
          definition = function(object){
            
            return(.generateModelFormulae(getBioFactors(object),
                                          getBatchFactors(object)))
          })

# ---- setModelFormula : ----

#' @name setModelFormula
#' @aliases setModelFormula,RflomicsMAE-method
#' @rdname generateModelFormulae
#' @param object an object of class \link{RflomicsSE} or 
#' class \link{RflomicsMAE-class}
#' @param modelFormula a string of model formula generated by 
#' \link{generateModelFormulae}
#' @return an object of \link{RflomicsSE} class or \link{RflomicsMAE-class}
#'  class
#' @description
#' \itemize{
#'    \item setModelFormula():
#'    Set the model formula stored in \code{metadata} slot}
setMethod(f          = "setModelFormula",
          signature  = "RflomicsMAE",
          definition = function(object, modelFormula=NULL){
            
            metadata(object)$design$Model.formula <- 
              paste(modelFormula, collapse = " ")
            
            for(name in names(object)){
              object[[name]] <- 
                setModelFormula(object[[name]], modelFormula)
            }
            return(object)
          })

#' @name setModelFormula
#' @aliases setModelFormula,RflomicsSE-method
#' @rdname generateModelFormulae
#' @exportMethod setModelFormula
setMethod(f          = "setModelFormula",
          signature  = "RflomicsSE",
          definition = function(object, modelFormula=NULL){
            
            metadata(object)$design$Model.formula <- 
              paste(modelFormula, collapse = " ")
            
            return(object)
          })

# ---- generateExpressionContrast ----
#' @title Contrast expressions
#' @name generateExpressionContrast
#' @aliases generateExpressionContrast,RflomicsSE-method
#' @rdname generateExpressionContrast
#' @description
#' Generate expression of contrasts based on chosed model formula.
#' \itemize{
#'    \item generateExpressionContrast:
#'  This function allows, from a model formulae, to give the 
#'  expression contrast data frames.
#'  Three types of contrasts are expressed:
#'  \itemize{
#'  \item pairwise comparison 
#'  \item averaged expression 
#'  \item interaction expression 
#' }}
#' @param object an object of class \link{RflomicsSE} or 
#' class \link{RflomicsMAE-class}
#' @param contrastType type of contrasts from which the possible 
#' contrasts are extracted ("average", "simple", "interaction"). 
#' Default is all contrasts types.
#' @return list of 1 or 3 data.frames of contrast expression
#' @exportMethod generateExpressionContrast
#' @author Christine Paysant-Le Roux, adapted by Nadia Bessoltane
#' @example inst/examples/statSetting.R
setMethod(f          = "generateExpressionContrast",
          signature  = "RflomicsSE",
          definition = function(object, contrastType=NULL){
            
            contrastTypes <- c("simple", "averaged", "interaction")
            
            modelFormula <- getModelFormula(object)
            # check
            if (is.null(modelFormula)) 
              stop("model formula is mandatory.")
            
            if (is(modelFormula, "formula"))
              modelFormula <- 
              paste(as.character(modelFormula), collapse = " ")
            
            # args for getExpressionContrastF()
            factorBio <- getBioFactors(object)
            ExpDesign <- getDesignMat(object)
            
            Contrasts.List  <-  
              .getExpressionContrastF(ExpDesign,  
                                      factorBio,
                                      modelFormula=modelFormula)
            
            if(is.null(contrastType))
              return(Contrasts.List)
            
            if(!contrastType %in% contrastTypes)
              stop("The contrastType argument must be one of the following values: ", 
                   contrastTypes)
            
            return(Contrasts.List[[contrastType]])
          })

#' @rdname generateExpressionContrast
#' @name generateExpressionContrast
#' @aliases generateExpressionContrast,RflomicsMAE-method
#' @exportMethod generateExpressionContrast
setMethod(f          = "generateExpressionContrast",
          signature  = "RflomicsMAE",
          definition = function(object, contrastType=NULL){
            
            contrastTypes <- c("simple", "averaged", "interaction")
            
            modelFormula <- getModelFormula(object)
            # check
            if (is.null(modelFormula)) 
              stop("model formula is mandatory.")
            
            if (is(modelFormula, "formula"))
              modelFormula <- 
              paste(as.character(modelFormula), collapse = " ")
            
            # args for getExpressionContrastF()
            factorBio <- getBioFactors(object)
            ExpDesign <- getDesignMat(object)
            
            Contrasts.List  <- 
              .getExpressionContrastF(ExpDesign, factorBio, modelFormula)
            
            if(is.null(contrastType))
              return(Contrasts.List)
            
            if(!contrastType %in% contrastTypes)
              stop("The contrastType argument must be one of the following values: ", 
                   contrastTypes)
            
            
            return(Contrasts.List[[contrastType]])
          })


# ---- setSelectedContrasts ----
#' @rdname generateExpressionContrast
#' @description
#' \itemize{
#'    \item setSelectedContrasts: 
#'  Set the selected contrasts stored in \code{metadata} slot}
#' @param contrastList a data.frame of contrasts generated by 
#' \link{generateExpressionContrast}
#' @return an object of \link{RflomicsSE} class or \link{RflomicsMAE-class}
#'  class
#' @exportMethod setSelectedContrasts
#' @name setSelectedContrasts
#' @aliases setSelectedContrasts,RflomicsMAE-method
setMethod(
  f          = "setSelectedContrasts",
  signature  = "RflomicsMAE",
  definition = function(object, contrastList=NULL){
    
    metadata(object)$design$Contrasts.Sel <- 
      updateSelectedContrasts(object, contrastList)
    
    # for each dataset
    for(SE.name in names(object)){
      
      object[[SE.name]] <- 
        setSelectedContrasts(object[[SE.name]], contrastList)
    }
    
    return(object)
  })


#' @rdname generateExpressionContrast
#' @exportMethod setSelectedContrasts
#' @name setSelectedContrasts
#' @aliases setSelectedContrasts,RflomicsSE-method
setMethod(f          = "setSelectedContrasts",
          signature  = "RflomicsSE",
          definition = function(object, contrastList=NULL){
            
            metadata(object)$design$Contrasts.Sel <- 
              updateSelectedContrasts(object, contrastList)
            
            return(object)
          })

# ---- getModelFormula : Get Model Formula : ----
#' @rdname generateModelFormulae
#' @name getModelFormula
#' @aliases getModelFormula,RflomicsMAE-method
#' @description
#' \itemize{
#'    \item getModelFormula:
#'  Access to the model formula of the statistical analysis}
#' @exportMethod getModelFormula
setMethod(f          = "getModelFormula",
          signature  = "RflomicsMAE",
          definition = function(object){
            
            return(metadata(object)$design$Model.formula)
          })

#' @exportMethod getModelFormula
#' @rdname generateModelFormulae
#' @name getModelFormula
#' @aliases getModelFormula,RflomicsSE-method
setMethod(f          = "getModelFormula",
          signature  = "RflomicsSE",
          definition = function(object){
            
            return(metadata(object)$design$Model.formula)
          })


# ---- getSelectedContrasts : ----
#' @rdname generateExpressionContrast
#' @exportMethod getSelectedContrasts
#' @name getSelectedContrasts
#' @aliases getSelectedContrasts,RflomicsMAE-method
#' @description
#' \itemize{
#'    \item getSelectedContrasts:
#'  List the selected contrasts}
setMethod(f          = "getSelectedContrasts",
          signature  = "RflomicsMAE",
          definition = function(object){
            
            return(metadata(object)$design$Contrasts.Sel)
          })


#' @rdname generateExpressionContrast
#' @exportMethod getSelectedContrasts
#' @name getSelectedContrasts
#' @aliases getSelectedContrasts,RflomicsSE-method
setMethod(f          = "getSelectedContrasts",
          signature  = "RflomicsSE",
          definition = function(object){
            
            return(metadata(object)$design$Contrasts.Sel)
          })

# ---- getContrastMatrix ----
# to write