% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/QRscore.R
\name{QRscoreTest}
\alias{QRscoreTest}
\title{QRscore Test}
\usage{
QRscoreTest(
  samples,
  labels = NULL,
  size_factors = NULL,
  p = NULL,
  wList = NULL,
  alternative = "two.sided",
  approx = "resample",
  type = "unbiased",
  n_mom = 100,
  resamp_number = 5000,
  zero_inflation = TRUE,
  LR.test = FALSE,
  pi_threshold = 0.95,
  gene.name = NULL,
  measure = "mean",
  perturb = TRUE,
  use_base_r = TRUE
)
}
\arguments{
\item{samples}{A numeric vector containing all sample measurements.}

\item{labels}{An optional vector of group labels corresponding to each entry
in \code{samples}.}

\item{size_factors}{Optional vector of size factors for weight estimation.
If provided, log-transformed size factors are used as offsets in
NB/ZINB model fitting.}

\item{p}{The exponent used in the power sum test statistic, required if
\code{wList} is not \code{NULL}.}

\item{wList}{An optional vector of weights; if \code{NULL}, weights are estimated
using an NB or ZINB model for multiple groups.}

\item{alternative}{Specifies the alternative hypothesis; must be one of
"two.sided", "greater", or "less".}

\item{approx}{The method used for p-value approximation, either "resample" or
"asymptotic".}

\item{type}{Specifies if the estimation of the p-value should be "unbiased",
"valid", or "both".}

\item{n_mom}{The number of moments to accompany the approximation, relevant
for moment-based methods.}

\item{resamp_number}{The number of resampling iterations used if \code{approx} is
"resample".}

\item{zero_inflation}{Indicates whether to account for zero inflation in
model-based weight estimation.}

\item{LR.test}{Whether a likelihood ratio test is used to decide between NB
and ZINB models.}

\item{pi_threshold}{Threshold for the proportion of zeros in ZINB models;
results in NA if exceeded.}

\item{gene.name}{Optional identifier for a gene, used in output messages.}

\item{measure}{Specifies the statistical measure to be analyzed ("mean" or
"dispersion") when weights are auto-generated.}

\item{perturb}{Boolean to indicate if data should be perturbed slightly to
prevent ties.}

\item{use_base_r}{Boolean to decide whether to use base R functions for
certain edge cases like Mann-Whitney tests.}
}
\value{
Returns the p-value of the test.
}
\description{
This function performs statistical tests on data from one or more groups
using summary statistics computed on weighted linear combinations of powers
of spacing statistics. It is capable of conducting one-sample tests,
two-sample tests, and multi-sample tests, utilizing either user-defined
weights or automatically generated weights based on Negative Binomial (NB)
or Zero-Inflated Negative Binomial (ZINB) models.
}
\details{
For one-sample tests, the function assesses the uniformity of data entries.
For two-sample and multi-sample tests, it evaluates whether groups are drawn
from the same distribution, with alternative hypotheses considering
differences in means or dispersions.

If the weights and \eqn{p} are given, the function calculates the test
statistic as: \deqn{||S_k||_{p,\boldsymbol{w}}^p=\sum_{j=1}^k
w_iS_{k}[j]^p} where \eqn{w_i} are weights, \eqn{x_i} are data points, and
\eqn{p} is the power specified.

In two-sample and multi-sample settings without specified weights, the
function can automatically estimate weights using score function for a
Negative Binomial or a Zero-Inflated Negative Binomial model, optimizing for
dispersion or mean shifts.
}
\examples{

set.seed(1)
# One-sample test example with normally distributed data
data <- abs(rnorm(10))
QRscoreTest(data,
    p = 2, wList = rep(1, 10), alternative = "two.sided",
    approx = "resample"
)

# Two-sample test with specified weights using normally distributed data
group1 <- rnorm(120, sd = 1)
group2 <- rnorm(120, sd = 2) # Different mean
data <- c(group1, group2)
labels <- c(rep("Group1", 120), rep("Group2", 120))
QRscoreTest(
    samples = data, labels = labels, p = 1, wList = c(60:0, seq_len(60)),
    alternative = "two.sided", approx = "resample"
)

# Two-sample test with automatically estimated weights from NB model
group1 <- rzinbinom(120, size = 2, mu = 20, pi = 0)
group2 <- rzinbinom(100, size = 2, mu = 30, pi = 0) # Different mean
data <- c(group1, group2)
labels <- c(rep("Group1", 120), rep("Group2", 100))
QRscoreTest(
    samples = data, labels = labels,
    approx = "asymptotic", measure = "mean", zero_inflation = FALSE
)

# Two-sample test with automatically estimated weights from ZINB model
group1 <- rzinbinom(100, size = 2, mu = 40, pi = 0.1)
group2 <- rzinbinom(200, size = 1, mu = 40, pi = 0.1)
data <- c(group1, group2)
labels <- c(rep("Group1", 100), rep("Group2", 200))
QRscoreTest(
    samples = data, labels = labels, alternative = "two.sided",
    approx = "asymptotic", measure = "dispersion"
)

# Three-sample test with automatically estimated weights from NB model
group1 <- rzinbinom(150, size = 1, mu = 30, pi = 0.1)
group2 <- rzinbinom(100, size = 2, mu = 30, pi = 0.1)
group3 <- rzinbinom(30, size = 3, mu = 30, pi = 0.1)
data <- c(group1, group2, group3)
labels <- c(rep("Group1", 150), rep("Group2", 100), rep("Group3", 30))
QRscoreTest(
    samples = data, labels = labels, alternative = "two.sided",
    approx = "asymptotic", measure = "dispersion"
)

# Two-sample NB test with size factors
sf <- runif(200, 0.5, 1.5)
x <- rnbinom(100, size = 2, mu = sf[1:100] * 20)
y <- rnbinom(100, size = 2, mu = sf[101:200] * 35)
QRscoreTest(
    samples = c(x, y), labels = rep(c("A", "B"), each = 100),
    size_factors = sf, measure = "mean", zero_inflation = FALSE
)
}
