% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/functions-MSnExp.R
\name{combineSpectraMovingWindow}
\alias{combineSpectraMovingWindow}
\title{Combine signal from consecutive spectra of LCMS experiments}
\usage{
combineSpectraMovingWindow(
  x,
  halfWindowSize = 1L,
  intensityFun = base::mean,
  mzd = NULL,
  timeDomain = FALSE,
  weighted = FALSE,
  ppm = 0,
  BPPARAM = bpparam()
)
}
\arguments{
\item{x}{\code{MSnExp} or \code{OnDiskMSnExp} object.}

\item{halfWindowSize}{\code{integer(1)} with the half window size for the moving
window.}

\item{intensityFun}{\code{function} to aggregate the intensity values per m/z
group. Should be a function or the name of a function. The function is
expected to return a \code{numeric(1)}.}

\item{mzd}{\code{numeric(1)} defining the maximal m/z difference below which
mass peaks are considered to represent the same ion/mass peak.
Intensity values for such grouped mass peaks are aggregated. If not
specified this value is estimated from the distribution of differences
of m/z values from the provided spectra (see details).}

\item{timeDomain}{\code{logical(1)} whether definition of the m/z values to be
combined into one m/z is performed on m/z values
(\code{timeDomain = FALSE}) or on \code{sqrt(mz)} (\code{timeDomain = TRUE}).
Profile data from TOF MS instruments should be aggregated based
on the time domain (see details). Note that a pre-defined \code{mzd} should
also be estimated on the square root of m/z values if
\code{timeDomain = TRUE}.}

\item{weighted}{\code{logical(1)} whether m/z values per m/z group should be
aggregated with an intensity-weighted mean. The default is to report
the mean m/z.}

\item{ppm}{\code{numeric(1)} to define an m/z relative deviation. Note that if
only \code{ppm} should be considered but not \code{mzd}, \code{mzd} should be set to
\code{0} (i.e. \code{mzd = 0}). This parameter is directly passed to
\code{\link[=meanMzInts]{meanMzInts()}}.}

\item{BPPARAM}{parallel processing settings.}
}
\value{
\code{MSnExp} with the same number of spectra than \code{x}.
}
\description{
\code{combineSpectraMovingWindow} combines signal from consecutive spectra within
a file. The resulting \code{MSnExp} has the same total number of spectra than the
original object, but with each individual's spectrum information
representing aggregated data from the original spectrum and its neighboring
spectra. This is thus equivalent with a smoothing of the data in retention
time dimension.

Note that the function returns always a \code{MSnExp} object, even if \code{x} was an
\code{OnDiskMSnExp} object.
}
\details{
The method assumes same ions being measured in consecutive scans (i.e. LCMS
data) and thus combines their signal which can increase the increase the
signal to noise ratio.

Intensities (and m/z values) for signals with the same m/z value in
consecutive scans are aggregated using the \code{intensityFun}.
m/z values of intensities from consecutive scans will never be exactly
identical, even if they represent signal from the same ion. The function
determines thus internally a similarity threshold based on differences
between m/z values within and between spectra below which m/z values are
considered to derive from the same ion. For robustness reasons, this
threshold is estimated on the 100 spectra with the largest number of
m/z - intensity pairs (i.e. mass peaks).

See \code{\link[=meanMzInts]{meanMzInts()}} for details.

Parameter \code{timeDomain}: by default, m/z-intensity pairs from consecutive
scans to be aggregated are defined based on the square root of the m/z
values. This is because it is highly likely that in all QTOF MS instruments
data is collected based on a timing circuit (with a certain variance) and
m/z values are later derived based on the relationship \code{t = k * sqrt(m/z)}.
Differences between individual m/z values will thus be dependent on the
actual m/z value causing both the difference between m/z values and their
scattering being different in the lower and upper m/z range. Determining
m/z values to be combined on the \code{sqrt(mz)} reduces this dependency. For
non-QTOF MS data \code{timeDomain = FALSE} might be used instead.
}
\note{
The function has to read all data into memory for the spectra combining
and thus the memory requirements of this function are high, possibly
preventing its usage on large experimental data. In these cases it is
suggested to perform the combination on a per-file basis and save the
results using the \code{\link[=writeMSData]{writeMSData()}} function afterwards.
}
\examples{

library(MSnbase)
library(msdata)

## Read a profile-mode LC-MS data file.
fl <- dir(system.file("sciex", package = "msdata"), full.names = TRUE)[1]
od <- readMSData(fl, mode = "onDisk")

## Subset the object to the retention time range that includes the signal
## for proline. This is done for performance reasons.
rtr <- c(165, 175)
od <- filterRt(od, rtr)

## Combine signal from neighboring spectra.
od_comb <- combineSpectraMovingWindow(od)

## The combined spectra have the same number of spectra, same number of
## mass peaks per spectra, but the signal is larger in the combined object.
length(od)
length(od_comb)

peaksCount(od)
peaksCount(od_comb)

## Comparing the chromatographic signal for proline (m/z ~ 116.0706)
## before and after spectra data combination.
mzr <- c(116.065, 116.075)
chr <- chromatogram(od, rt = rtr, mz = mzr)
chr_comb <- chromatogram(od_comb, rt = rtr, mz = mzr)

par(mfrow = c(1, 2))
plot(chr)
plot(chr_comb)
## Chromatographic data is "smoother" after combining.
}
\seealso{
\code{\link[=meanMzInts]{meanMzInts()}} for the function combining spectra provided in
a \code{list}.

\code{\link[=estimateMzScattering]{estimateMzScattering()}} for a function to estimate m/z value scattering in
consecutive spectra.
}
\author{
Johannes Rainer, Sigurdur Smarason
}
