#' INSPEcT package
#'
#' @description
#' INSPEcT (INference of Synthesis, Processing and dEgradation rates from Transcriptomic data),
#' is a package that analyse RNA-seq data in order to evaluate synthesis, processing and degradation
#' rates and asses via modeling the rates that determines changes in RNA levels.
#' 
#' To see how the typical workflow of INSPEcT works, type: \cr
#' \code{vignette('INSPEcT')}
#'
#' INSPEcT implements two main classes (\code{\linkS4class{INSPEcT}} and 
#' \code{\linkS4class{INSPEcT_diffsteady}}) and their corresponding methods. 
#' To have a detailed description of how the two classes are structured 
#' and which methods apply on, type:
#' 
#' \code{?'INSPEcT-class'} \cr
#' \code{?'INSPEcT_diffsteady-class'}
#'
#' 
#' To obtain the citation, type: \cr
#' \code{citation('INSPEcT')}
#'
#'
#' @docType package
#' @name INSPEcT
NULL

#' An S4 class to represent models generated by INSPEcT
#'
#' @description 
#' INSPEcT_model is a class able to store all the results of the modeling of synthesis, processing and degradation rates made via the method \code{\link{modelRates}} (slot ratesSpecs).
#' It also stores the criteria (slot parameter) to choose between the many models tested for each gene the one that better describes the data and the results. 
#' The slot simple is a flag that distinguish wheter the model contains the information of the introns or not.
#' In case not, the flag simple is set to TRUE. 
#' Also the method \code{\link{makeSimModel}} of class \code{\link{INSPEcT-class}} creates an object of class INSPEcT_model. 
#' This object will be used by \code{\link{makeSimDataset}} to generate a complete simulated data-set, whose classification performance can be tested.
#' @slot params A list that defines thresholds and how to perform log likelihood ratio tests
#' @slot ratesSpecs A list containing the modeling output
#' @slot simple A logical that indicates whether the mode of INSPEcT is simple (no pre-mRNA and degradation rates) or not.
#' @details Methods that apply to INSPEcT_model class are
#'	\code{\link[=Extract]{[}} \cr
#'	\code{\link{AIC}} \cr
#'	\code{\link{chisqtest}} \cr
#'	\code{\link{correlationPlot}} \cr
#'	\code{\link{geneClass}} \cr
#'	\code{\link{logLik}} \cr
#'	\code{\link{makeModelRates}} \cr
#'	\code{\link{makeSimDataset}} \cr
#'	\code{\link{modelSelection}} \cr
#'	\code{\link{rocCurve}} \cr
#'	\code{\link{rocThresholds}} \cr
#'	\code{\link{show}} \cr
setClass('INSPEcT_model', 
	slots=c(
		params='list'
		, ratesSpecs='list'
		, simple='logical'
		)
	, prototype=list(
		simple=FALSE
			)
		)

# class definition
#' An S4 class to contain all rates and models generated by INSPEcT
#'
#' @description 
#' INSPEcT is a class able to store all the estimated rates and concentrations (slot ratesFirstGuess), 
#' the modeled rates and concentrations (slot modelRates) and the model themselves (slot model).
#' Within the class INSPEcT other information regarding the experimental design are stored, such as the time
#' points where experiments were performed (slot tpts) and, if provided, the nascent RNA collecting time (slot tL)
#' and the normalization scale fators used for nascent (labeledSF) RNA-seq libraries.
#' A list of parameters that will be used during the 
#' modeling process is stored within the slot params and can be accessed by \code{\link{modelingParams}}.
#' A new instance of the class INSPEcT can be generated by the constructor function \code{\link{newINSPEcT}}.
#' @slot params A list of parameters of the modeling part
#' @slot ratesFirstGuess An object of class ExpressionSet that contains all the rates and concentrations guessed from the first part of INSPEcT analysis (before modeling)
#' @slot ratesFirstGuessVar An object of class ExpressionSet that contains the variances related to rates and concentrations guessed from the first part of INSPEcT analysis (before modeling)
#' @slot confidenceIntervals An object of class ExpressionSet that contains the confidence intervals.
#' @slot model An object of class INSPEcT_model that contains the output of the mdoeling.
#' @slot modelRates An object of class ExpressionSet that contains all modeled the rates and concentrations.
#' @slot ratePvals A matrix containing the p-value relative to the variability of synthesis, processing and degradation for each gene.
#' @slot tpts A numeric vector of the time-points.
#' @slot labeledSF A numeric vector of the scaling factor used for inter time-point normalization of Nascent-seq libraries.
#' @slot tL A numeric containing the length of the Nascent pulse.
#' @slot NoNascent A logical indicating if the nascent RNA was included into the analysis.
#' @slot NF A logical indicating if the modeling approach is Non-Functional
#' @slot degDuringPulse A logical indicating if degradation of RNA during the 4sU pulse was considered.
#' @slot version A character indicating the version of INSPEcT that created the object
#' @details Methods that apply to INSPEcT class are
#'	\code{\link{AIC}} \cr
#'	\code{\link[=Extract]{[}} \cr
#'	\code{\link{calculateDelta}} \cr
#'	\code{\link{calculateRatePvals}} \cr
#'	\code{\link{calculateTau}} \cr
#'	\code{\link{chisqmodel}} \cr
#'	\code{\link{chisqtest}} \cr
#'	\code{\link{combine}} \cr
#'	\code{\link{compareSteady}} \cr
#'	\code{\link{compareSteadyNoNascent}} \cr
#'	\code{\link{computeConfidenceIntervals}} \cr
#'	\code{\link{correlationPlot}} \cr
#'	\code{\link{dim}} \cr
#'	\code{\link{featureNames}} \cr
#'	\code{\link{geneClass}} \cr
#'	\code{\link{inHeatmap}} \cr
#'	\code{\link{labeledSF}} \cr
#'	\code{\link{logLik}} \cr
#'	\code{\link{makeModelRates}} \cr
#'	\code{\link{makeOscillatorySimModel}} \cr
#'	\code{\link{makeSimModel}} \cr
#'	\code{\link{modelRates}} \cr
#'	\code{\link{modelRatesNF}} \cr
#'	\code{\link{modelSelection}} \cr
#'	\code{\link{modelingParams}} \cr
#'	\code{\link{nGenes}} \cr
#'	\code{\link{nTpts}} \cr
#'	\code{\link{plotGene}} \cr
#'	\code{\link{processingDelay}} \cr
#'	\code{\link{ratePvals}} \cr
#'	\code{\link{ratesFirstGuess}} \cr
#'	\code{\link{ratesFirstGuessVar}} \cr
#'	\code{\link{removeModel}} \cr
#'	\code{\link{rocCurve}} \cr
#'	\code{\link{rocThresholds}} \cr
#'	\code{\link{setConfidenceIntervals}} \cr
#'	\code{\link{show}} \cr
#'	\code{\link{split}} \cr
#'	\code{\link{tpts}} \cr
#'	\code{\link{viewConfidenceIntervals}} \cr
#'	\code{\link{viewModelRates}} \cr
setClass('INSPEcT', 
				 slots=c(
				 	params='list'
				 	, ratesFirstGuess='ExpressionSet'
				 	, ratesFirstGuessVar='ExpressionSet'
				 	, confidenceIntervals='ExpressionSet'
				 	, model='INSPEcT_model'
				 	, modelRates='ExpressionSet'
				 	, ratePvals='data.frame'
				 	, tpts='vector'
				 	, labeledSF='numeric'
				 	, tL='numeric'
				 	, NoNascent='logical'
				 	, NF='logical'
				 	, degDuringPulse='logical'
				 	, version='character'
				 )
				 , prototype=list(
				 	NoNascent=FALSE,
				 	NF=FALSE,
				 	degDuringPulse=FALSE,
				 	version=as.character(packageVersion('INSPEcT'))
				 )
)

#' An S4 class to represent comparisons between two steady-state conditions
#'
#' @description
#' INSPEcT_diffsteady is a class able to store the results of the comparisons between two steady states.
#' An object of class INSPEcT_diffsteady is created with the method "compareSteady" applied on 
#' two "INSPEcT" objects (see \code{\link{compareSteady}}).
#' @slot synthesis A data.frame which contains both input data and comparisons results regarding
#' synthesis rates
#' @slot degradation A data.frame which contains both input data and comparisons results regarding
#' degradation rates
#' @slot processing A data.frame which contains both input data and comparisons results regarding
#' processing rates
#' @slot modeling_res A data.frame which contains modeling results
#' @details
#' Methods associated to the class INSPEcT_diffsteady are:
#' \itemize{
#'   \item synthesis: Accessor to the synthesis rates and their comparisons.
#'   \item degradation: Accessor to the degradation rates and their comparisons.
#'   \item processing: Accessor to the processing rates and their comparisons.
#'	 \item plotMA: visualization fuction for rates comparisons, see \code{\link{plotMA}}
#' }
setClass('INSPEcT_diffsteady', 
	slots=c(
		synthesis='data.frame'
		, degradation='data.frame'
		, processing='data.frame'
		, modeling_res='data.frame'
		)
	)

###### add the description of the INSPEcT_steadyNoNascent-class ##########

#' An S4 class to represent steady-state analysis without nascent RNA
#'
#' @description
#' INSPEcT_steadyNoNascent is a class able to store data and arguments that are necessary to 
#' make the analysis concerning premature and mature expressions in different samples. In 
#' particular, the ratio between mature and premature can be calculated, which reflects
#' the ratio between the rates of processing and degradation in individaul genes 
#' (see \code{\link{PTratio}}), and the analysis of post-transcriptionally regualted genes
#' can be run to identify genes that in specfic samples show a trand which cannot be attributed
#' to transcriptional regulation alone (see \code{\link{PTreg}}).
#' @slot sampleNames Vector with the names of the samples (columns of the dataset)
#' @slot geneNames Vector with the names of the genes (rows of the dataset)
#' @slot premature Matrix containing the expressions of the premature RNAs (row=genes, columns=samples)
#' @slot mature Matrix containing the expressions of the emature RNAs (row=genes, columns=samples)
#' @slot prematureVar Matrix containing the expressions variances of the premature RNAs (row=genes, columns=samples)
#' @slot matureVar Matrix containing the expressions variances of the emature RNAs (row=genes, columns=samples)
#' @slot trivialAngle Numeric that indicates the angle (slope) of the linear model between mature and premature expressions
#' @slot log2FCThreshold Numeric that describes the threshold of the variation to be considered significant
#' @slot expressionThreshold Numeric that describes the threshold of the expression to consider the gene expressed
#' @slot referenceCondition A sample identifier that set the reference for the post-transcriptional regulation analysis, 
#' if NULL the median of all samples is used
#' @slot ptreg Matrix containing the post-transcriptioanl regulation state of each gene in the different samples (row=genes, columns=samples)
setClass('INSPEcT_steadyNoNascent', 
				 slots=c(
				 	sampleNames='character',
				 	geneNames='character',
				 	premature='matrix',
				 	mature='matrix',
				 	prematureVar='matrix',
				 	matureVar='matrix',
				 	trivialAngle='numeric',
				 	log2FCThreshold='numeric',
				 	expressionThreshold='numeric',
				 	referenceCondition='character',
				 	ptreg='matrix'
				 )
)