#' @importClassesFrom GenomicRanges GRanges
#' @importClassesFrom InteractionSet GInteractions

setClassUnion("GRangesOrGInteractions", members = c("GRanges", "GInteractions"))
setClassUnion("characterOrNULL", members = c("character", "NULL"))
setClassUnion("numericOrNULL", members = c("numeric", "NULL"))

#' @title `HiCExperiment` S4 class
#' 
#' @name HiCExperiment
#' @rdname HiCExperiment
#' 
#' @description
#' 
#' The `HiCExperiment` class describes Hi-C contact files imported in R, either 
#' through the `HiCExperiment` constructor function or using the `import` 
#' method implemented by `HiCExperiment` package. 
#'
#' @slot fileName Path of Hi-C contact file
#' @slot focus Chr. coordinates for which interaction counts are extracted 
#'   from the Hi-C contact file.
#' @slot resolutions Resolutions available in the Hi-C contact file.
#' @slot resolution Current resolution
#' @slot interactions Genomic Interactions extracted from the Hi-C contact file
#' @slot scores Available interaction scores. 
#' @slot topologicalFeatures Topological features associated with the dataset 
#'   (e.g. loops (\<GInteractions\>), borders (\<GRanges\>), 
#'   viewpoints (\<GRanges\>), etc...)
#' @slot pairsFile Path to the .pairs file associated with the Hi-C contact file
#' @slot metadata metadata associated with the Hi-C contact file.
#' 
#' @param file CoolFile or plain path to a Hi-C contact file
#' @param resolution Resolution to use with the Hi-C contact file
#' @param focus Chromosome coordinates for which 
#'   interaction counts are extracted from the Hi-C contact file, provided
#'   as a character string (e.g. "II:4001-5000"). If not provided, 
#'   the entire Hi-C contact file will be imported. 
#' @param metadata list of metadata
#' @param topologicalFeatures topologicalFeatures provided as a named SimpleList
#' @param pairsFile Path to an associated .pairs file (optional)
#' @param bed Path to regions file generated by HiC-Pro (optional)
#' @param gi GInteractions object
#' 
#' @return An `HiCExperiment` object.
#' 
#' @importFrom methods setClass
#' @importFrom S4Vectors metadata
#' @importClassesFrom S4Vectors Annotated
#' @importMethodsFrom S4Vectors metadata
#' @seealso [AggrHiCExperiment()], [CoolFile()], [HicFile()], [HicproFile()], [PairsFile()]
#' @examples 
#' #####################################################################
#' ## Create a HiCExperiment object from a disk-stored contact matrix ##
#' #####################################################################
#' 
#' mcool_file <- HiContactsData::HiContactsData("yeast_wt", "mcool")
#' pairs_file <- HiContactsData::HiContactsData("yeast_wt", "pairs.gz")
#' contacts <- HiCExperiment(
#'     file = mcool_file, 
#'     resolution = 8000L, 
#'     pairsFile = pairs_file
#' )
#' contacts
#' 
#' #####################################################################
#' ## ----- Manually create a HiCExperiment from GInteractions ------ ##
#' #####################################################################
#' 
#' gis <- interactions(contacts)[1:1000]
#' contacts2 <- makeHiCExperimentFromGInteractions(gis)
#' contacts2
#' 
#' #####################################################################
#' ## -------- Slots present in an HiCExperiment object ------------- ##
#' #####################################################################
#' 
#' fileName(contacts)
#' focus(contacts)
#' resolutions(contacts)
#' resolution(contacts)
#' interactions(contacts)
#' scores(contacts)
#' topologicalFeatures(contacts)
#' pairsFile(contacts)
#' 
#' #####################################################################
#' ## ---------------------- Slot getters --------------------------- ##
#' #####################################################################
#' 
#' scores(contacts, 1) |> head()
#' scores(contacts, 'balanced') |> head()
#' topologicalFeatures(contacts, 1)
#' 
#' #####################################################################
#' ## ---------------------- Slot setters --------------------------- ##
#' #####################################################################
#' 
#' scores(contacts, 'random') <- runif(length(contacts))
#' topologicalFeatures(contacts, 'loops') <- InteractionSet::GInteractions(
#'   GenomicRanges::GRanges('II:15324'), 
#'   GenomicRanges::GRanges('II:24310')
#' )
#' pairsFile(contacts) <- HiContactsData('yeast_wt', 'pairs.gz')
#' 
#' #####################################################################
#' ## ------------------ Subsetting functions ----------------------- ##
#' #####################################################################
#' 
#' contacts[1:100]
#' contacts['II']
#' contacts[c('II', 'III')]
#' contacts['II|III']
#' contacts['II:10001-30000|III:50001-90000']
#' 
#' #####################################################################
#' ## --------------------- Utils functions ------------------------- ##
#' #####################################################################
#' ## Adapted from other packages
#' 
#' seqinfo(contacts)
#' bins(contacts)
#' anchors(contacts)
#' regions(contacts)
#' 
#' #####################################################################
#' ## ------------- Coercing HiCExperiment objects ------------------ ##
#' #####################################################################
#' 
#' as(contacts, 'GInteractions')
#' as(contacts, 'ContactMatrix')
#' as(contacts, 'matrix')[seq_len(10), seq_len(10)]
#' as(contacts, 'data.frame')[seq_len(10), seq_len(10)]
NULL

#' @rdname HiCExperiment
#' @export

methods::setClass("HiCExperiment", 
    contains = c("Annotated"), 
    slots = c(
        fileName = "character",
        focus = "characterOrNULL", 
        resolutions = "numeric", 
        resolution = "numeric", 
        interactions = "GInteractions",
        scores = "SimpleList", 
        topologicalFeatures = "SimpleList",
        pairsFile = "characterOrNULL",
        metadata = "list"
    )
)

#' @rdname HiCExperiment
#' @export

HiCExperiment <- function(
    file, 
    resolution = NULL, 
    focus = NULL, 
    metadata = list(), 
    topologicalFeatures = S4Vectors::SimpleList(
        'compartments' = GenomicRanges::GRanges(), 
        'borders' = GenomicRanges::GRanges(), 
        'loops' = InteractionSet::GInteractions(
            GenomicRanges::GRanges(), 
            GenomicRanges::GRanges()
        ), 
        'viewpoints' = GenomicRanges::GRanges()
    ), 
    pairsFile = NULL, 
    bed = NULL
) {
    file <- gsub('~', Sys.getenv('HOME'), file)
    stopifnot(file.exists(file))
    if (!is.null(resolution)) resolution <- as.integer(resolution)
    
    if (.is_cool(file) | .is_mcool(file)) {
        return(.HiCExperimentFromCoolFile(
            file = file,
            resolution = resolution,
            focus = focus,
            metadata = metadata,
            topologicalFeatures = topologicalFeatures,
            pairsFile = pairsFile
        ))
    }
    if (.is_hic(file)) {
        return(.HiCExperimentFromHicFile(
            file = file,
            resolution = resolution,
            focus = focus,
            metadata = metadata,
            topologicalFeatures = topologicalFeatures,
            pairsFile = pairsFile
        ))
    }
    if (.is_hicpro_matrix(file)) {
        if (is.null(bed)) stop("Regions files not provided.")
        if (!is.null(resolution)) stop("Resolution cannot be specified when importing HiC-Pro files.")
        if (!is.null(focus)) stop("Focus cannot be specified when importing HiC-Pro files.")
        return(.HiCExperimentFromHicproFile(
            file = file,
            bed = bed,
            metadata = metadata,
            topologicalFeatures = topologicalFeatures,
            pairsFile = pairsFile
        ))
    }
    return(NA)
}

setValidity("HiCExperiment",
    function(object) {
        if (!is(focus(object), "characterOrNULL"))
            return("'focus' slot must be a characterOrNULL")
        if (!is(resolutions(object), "numeric"))
            return("'resolutions' slot must be an numeric vector")
        if (!is(scores(object), "SimpleList"))
            return("'scores' slot must be a SimpleList")
        TRUE
    }
)

#' @rdname HiCExperiment
#' @export

makeHiCExperimentFromGInteractions <- function(gi) {
    resolution <- GenomicRanges::width(InteractionSet::regions(gi))[1]
    mcols <- S4Vectors::mcols(gi) |> 
        as.data.frame()
    if ('bin_id1' %in% colnames(mcols)) mcols[,'bin_id1'] <- NULL
    if ('bin_id2' %in% colnames(mcols)) mcols[,'bin_id2'] <- NULL
    x <- methods::new("HiCExperiment", 
        fileName = "",
        focus = "", 
        resolutions = resolution, 
        resolution = resolution, 
        interactions = gi, 
        scores = S4Vectors::SimpleList(as.list(mcols)), 
        topologicalFeatures = S4Vectors::SimpleList(), 
        pairsFile = NULL, 
        metadata = list()
    )
    methods::validObject(x)
    return(x)
}

.HiCExperimentFromCoolFile <- function(
    file, 
    resolution = NULL, 
    focus = NULL, 
    metadata = list(), 
    topologicalFeatures = S4Vectors::SimpleList(
        'compartments' = GenomicRanges::GRanges(), 
        'borders' = GenomicRanges::GRanges(), 
        'loops' = InteractionSet::GInteractions(
            GenomicRanges::GRanges(), 
            GenomicRanges::GRanges()
        ), 
        'viewpoints' = GenomicRanges::GRanges()
    ), 
    pairsFile = NULL
) {
    
    ## -- Check that provided file is valid
    .check_cool_file(file)
    .check_cool_format(file, resolution)

    ## -- Read interactions
    gis <- .cool2gi(file, resolution = resolution, coords = focus) |> sort()
    mcols <- GenomicRanges::mcols(gis)
    GenomicRanges::mcols(gis) <- mcols[, c('bin_id1', 'bin_id2')]

    ## -- Create contact object
    x <- methods::new("HiCExperiment", 
        fileName = as.character(file),
        focus = focus, 
        resolutions = .lsCoolResolutions(file), 
        resolution = ifelse(is.null(resolution), .lsCoolResolutions(file)[1], resolution), 
        interactions = gis, 
        scores = S4Vectors::SimpleList(
            'count' = as.numeric(mcols$count),
            'balanced' = as.numeric(mcols$score)
        ), 
        topologicalFeatures = topologicalFeatures, 
        pairsFile = pairsFile, 
        metadata = metadata
    )
    methods::validObject(x)
    return(x)
} 

.HiCExperimentFromHicFile <- function(
    file, 
    resolution = NULL, 
    focus = NULL, 
    metadata = list(), 
    topologicalFeatures = S4Vectors::SimpleList(
        'compartments' = GenomicRanges::GRanges(), 
        'borders' = GenomicRanges::GRanges(), 
        'loops' = InteractionSet::GInteractions(
            GenomicRanges::GRanges(), 
            GenomicRanges::GRanges()
        ), 
        'viewpoints' = GenomicRanges::GRanges()
    ), 
    pairsFile = NULL
) {
    
    ## -- Check that provided file is valid
    file <- gsub('~', Sys.getenv('HOME'), file)
    .check_hic_file(file)
    .check_hic_format(file, resolution)

    ## -- Read interactions
    gis <- .hic2gi(file, resolution = resolution, coords = focus) |> sort()
    mcols <- GenomicRanges::mcols(gis)
    GenomicRanges::mcols(gis) <- mcols[, c('bin_id1', 'bin_id2')]

    ## -- Create HiCExperiment
    if (!is.null(focus)) focus <- gsub("(:.*[^:]*):", "\\1-", focus)
    x <- methods::new("HiCExperiment", 
        fileName = as.character(file),
        focus = focus, 
        resolutions = rev(strawr::readHicBpResolutions(file)), 
        resolution = resolution, 
        interactions = gis, 
        scores = S4Vectors::SimpleList(
            'count' = as.numeric(mcols$count),
            'balanced' = as.numeric(mcols$score)
        ), 
        topologicalFeatures = topologicalFeatures, 
        pairsFile = pairsFile, 
        metadata = metadata
    )
    methods::validObject(x)
    return(x)
} 

.HiCExperimentFromHicproFile <- function(
    file, 
    bed, 
    metadata = list(), 
    topologicalFeatures = S4Vectors::SimpleList(
        'compartments' = GenomicRanges::GRanges(), 
        'borders' = GenomicRanges::GRanges(), 
        'loops' = InteractionSet::GInteractions(
            GenomicRanges::GRanges(), 
            GenomicRanges::GRanges()
        ), 
        'viewpoints' = GenomicRanges::GRanges()
    ), 
    pairsFile = NULL
) {
    
    ## -- Check that provided file is valid
    file <- gsub('~', Sys.getenv('HOME'), file)
    .check_hicpro_files(file, bed)

    ## -- Read interactions
    gis <- .hicpro2gi(file, bed) |> sort()
    mcols <- GenomicRanges::mcols(gis)
    GenomicRanges::mcols(gis) <- mcols[, c('bin_id1', 'bin_id2')]
    res <- GenomicRanges::width(regions(gis))[[1]]

    ## -- Create HiCExperiment
    x <- methods::new("HiCExperiment", 
        fileName = as.character(file),
        focus = NULL, 
        resolutions = res, 
        resolution = res, 
        interactions = gis, 
        scores = S4Vectors::SimpleList(
            'count' = as.numeric(mcols$count)
        ), 
        topologicalFeatures = topologicalFeatures, 
        pairsFile = pairsFile, 
        metadata = c(list(regions = bed), metadata)
    )

    ## -- If imported data is all integers, run ICE and save `balanced`
    s <- scores(x, 'count')
    if (all(as.integer(s) == s)) {
        if (requireNamespace("HiContacts", quietly = TRUE)) {
            x <- HiContacts::normalize(x)
            names(x@scores)[2] <- 'balanced'
        }
        else {
            warning('Install `HiContacts` package (`BiocManager::install("HiContacts")`)\nto balance Hi-C data.')
        }
    }
    else {
        message("Imported data stored in `balanced` scores.")
        names(x@scores)[1] <- 'balanced'
    }

    methods::validObject(x)
    return(x)
} 
