% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/utils.R
\name{kinshipPairs}
\alias{kinshipPairs}
\title{Extract pairs of individuals matching certain kinship criteria}
\usage{
kinshipPairs(
  x,
  condition = function(x) x >= 0.25,
  duplicates = c("keep", "first", "last", "random"),
  id = NULL,
  family = NULL
)
}
\arguments{
\item{x}{A \code{FAData} object (or object inheriting from that).}

\item{condition}{A \code{function} defining how individuals should be selected
based on the object's kinship matrix. The default is to select all
individuals with a kinship \verb{>= 0.25}. Note that the diagonal of the
kinship matrix (i.e. the kinship of individuals with itself) is always
skipped, so no additional criteria is needed to avoid self-pairs.}

\item{duplicates}{\code{character(1)} defining how to deal with duplicated IDs
in the result returned by the function.
See function description and examples below for more details. Defaults
to \code{duplicates = "keep"} returning all pairs of IDs matching \code{condition}.}

\item{id}{optional identifiers of subsets of individuals on which the pairs
should be defined. Defaults to \code{id = NULL} hence the full data set is
considered.}

\item{family}{optional family identifiers if pairs should only defined for
selected families. Defaults to \code{family = NULL} hence the full data set is
considered.}
}
\value{
A two column \code{matrix} with the IDs (colnames/rownames of the kinship matrix
or as defined in \code{x$id}) of the pairs. If \code{duplicates} is either \code{"first"},
\code{"last"} or \code{"random"} each ID is only returned once (i.e. no ID is reported
more than one time).
}
\description{
The \code{kinshipPairs} function allows to extract pairs of individuals matching
a user-defined kinship \emph{condition} (e.g. individuals with a kinship larger
than 0.0625). Such sets of paired individuals (along with paired unrelated
values) would enable a \emph{familial resemblance} analysis on quantitative
traits (Ziegler 2010) (see examples below for details).

By default, \code{kinshipPairs} returns all pairs of individuals for which the
\code{condition} on the kinship matrix matches (e.g. all pairs of individuals with
a kinship coefficient larger than or equal to 0.25). Individuals can thus
be reported multiple times (see examples below). Parameter \code{duplicates} can
be used to define a strategy to avoid such duplicated IDs. Supported are:
\itemize{
\item \code{duplicates = "keep"}: the default, return all values.
\item \code{duplicates = "first"}: report only the first pair of individuals for each
individual ID.
\item \code{duplicates = "last"}: report only the last pair of individuals for each
individual ID.
\item \code{duplicates = "random"}: randomly select one pair of individuals for
each individual ID.
}

For any setting different than \code{duplicates = "keep"} each individual will
only be listed \strong{once} in the resulting matrix.
}
\examples{

##########################
##
##  Create a new FAData object
##
## Load the Minnesota Breast Cancer record and subset to the
## first families.
data(minnbreast)
mbsub <- minnbreast[minnbreast$famid \%in\% 1:20, ]
mbped <- mbsub[, c("famid", "id", "fatherid", "motherid", "sex")]
## Renaming column names
colnames(mbped) <- c("family", "id", "father", "mother", "sex")
## Defining the optional argument age.
Age <- mbsub$endage
names(Age) <- mbsub$id
## Create the object
fad <- FAData(pedigree=mbped, age=Age)

## Getting all pairs of individuals with a kinship coefficient >= 0.25
## keeping all duplicates
rel_pairs <- kinshipPairs(fad)
head(rel_pairs)
## As we see, we have multiple times the individual 1 etc.

## For an actual correlation analysis it would be better to drop duplicates.
## Below we randomly select individual pairs if they occurr multiple times
rel_pairs <- kinshipPairs(fad, duplicates = "random")
head(rel_pairs)

## In addition we extract pairs of individuals that are much less related.
## For this examples we consider all individuals with a kinship
## coefficient < 0.03125 (second cousin) to be *unrelated*.
unrel_pairs <- kinshipPairs(fad, duplicates = "random",
    condition = function(z) z < 0.03125)
head(unrel_pairs)

## For a familial resemblance analysis we can now calculate the correlation
## coefficient of a quantitative trait between pairs of related individuals
## and compare that with the correlation coefficient calculated on unrelated
## individuals. For our toy example we use the participant's age, since we
## don't have any other quantitative values available.
cor_rel <- cor(age(fad)[rel_pairs[, 1]], age(fad)[rel_pairs[, 2]],
    use = "pairwise.complete.obs")

cor_unrel <- cor(age(fad)[unrel_pairs[, 1]], age(fad)[unrel_pairs[, 2]],
    use = "pairwise.complete.obs")
cor_rel
cor_unrel

## We don't see a clear difference in the correlation, thus, the age (as
## expected) has no familial component.
}
\references{
Ziegler A., Koenig I. R. (2010). Familiality, Heristability, and Segregation
Analysis. In A Statistical Approach to Genetic Epidemiology: With Access to
E-Learning Platform by Friedrich Pahlke, Second Edition.
\doi{10.1002/9783527633654.ch6}.
}
\seealso{
\link{PedigreeUtils} for other pedigree utility functions.
}
\author{
Johannes Rainer
}
