% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calculate_spike_percentage.R
\name{calculate_spike_percentage}
\alias{calculate_spike_percentage}
\title{Calculate Spike Percentage for Specified Taxa in a Phyloseq or TSE Object}
\usage{
calculate_spike_percentage(
  obj,
  merged_spiked_species = NULL,
  merged_spiked_hashcodes = NULL,
  output_file = NULL,
  passed_range = c(0.1, 11)
)
}
\arguments{
\item{obj}{A \code{phyloseq} or \code{TreeSummarizedExperiment} object containing the microbial data.}

\item{merged_spiked_species}{A character vector of spiked taxa names (can be from any taxonomic level).}

\item{merged_spiked_hashcodes}{A character vector of spiked hashcodes (ASV/OTU IDs) to check in the dataset. Default is NULL.}

\item{output_file}{A character string specifying the path to save the output files. Default is NULL (no files are written).}

\item{passed_range}{A numeric vector of length 2 specifying the range of percentages to categorize results as "passed". Default is c(0.1, 11).}
}
\value{
A data frame with the following columns:
\describe{
\item{Sample}{Sample identifier.}
\item{Total_Reads}{Total number of reads in the sample.}
\item{Spiked_Reads}{Number of reads mapped to the spike-in taxa.}
\item{Percentage}{Percentage of spike-in reads (Spiked_Reads / Total_Reads * 100).}
\item{Result}{Quality control result, either \code{"passed"} or \code{"failed"}, based on the specified range.}
}
}
\description{
In spike-in based absolute quantitation workflows, the acceptable recovery range of spike-in reads
is system dependent, varying with sequencing platform, extraction protocol, and microbial community structure.
This function calculates the percentage of reads, categorizes the results as passed or failed,
optionally saves the results as DOCX and CSV files.It also visualizes the relationship between observed spike-in abundance and total reads,
stratified across user-defined recovery intervals, to empirically determine the optimal range for quality control.
}
\examples{
if (requireNamespace("DspikeIn", quietly = TRUE)) {
  # Load example phyloseq object
  data("physeq_16SOTU", package = "DspikeIn")

  # ----------- Phyloseq Example -----------
  species_name <- c("Tetragenococcus_halophilus", "Tetragenococcus_sp.")
  merged_spiked_species <- "Tetragenococcus_halophilus"

  # Pre-process the phyloseq object to merge spike-in taxa
  merged_physeq <- Pre_processing_species(
    physeq_16SOTU,
    species_name = species_name,
    merge_method = "sum"
  )

  # Perform spike-in percentage calculation
  output_docx <- file.path(tempdir(), "spike_summary_physeq.docx")
  result_physeq <- calculate_spike_percentage(
    obj = merged_physeq,
    merged_spiked_species = merged_spiked_species,
    output_file = output_docx,
    passed_range = c(0.1, 20)
  )
  print(result_physeq)

  # ----------- TSE Example -----------
  tse_16SOTU <- convert_phyloseq_to_tse(physeq_16SOTU)
  merged_tse <- Pre_processing_species(
    tse_16SOTU,
    species_name = species_name,
    merge_method = "sum"
  )

  output_docx_tse <- file.path(tempdir(), "spike_summary_tse.docx")
  result_tse <- calculate_spike_percentage(
    obj = merged_tse,
    merged_spiked_species = merged_spiked_species,
    output_file = output_docx_tse,
    passed_range = c(0.1, 20)
  )
  print(result_tse)

  # Clean up temporary files
  if (file.exists(output_docx)) unlink(output_docx, force = TRUE)
  if (file.exists(output_docx_tse)) unlink(output_docx_tse, force = TRUE)
}
}
\seealso{
\code{\link{Pre_processing_species}}, \code{\link{calculate_spike_percentage}}
}
