#' Density-based clustering of spectral similarity data
#'
#' \code{OPTICStbl()} performs density-based clustering
#' of spectral similarity data using the OPTICS algorithm.
#'
#' @param distmat A distance matrix as generated by
#' \code{\link{distanceMatrix}}.
#'
#' @param eps,minPts OPTICS parameters, see \code{\link[dbscan]{optics}}.
#'
#' @param eps_cl The reachability distance used for cluster determination,
#' see \code{\link[dbscan:optics]{extractDBSCAN}}.
#'
#' @return A \code{data.frame} with feature name, cluster ID and
#' OPTICS order for each feature in \code{distmat}.
#'
#' @details The function internally uses \code{\link[dbscan]{optics}}
#' and \code{\link[dbscan:optics]{extractDBSCAN}} from the \pkg{dbscan}
#' package.
#'
#' @seealso OPTICSplot
#'
#' @importFrom stats as.dist
#'
#' @import dbscan
#'
#' @examples
#' load(file = system.file("extdata",
#'     "distmat.RData",
#'     package = "CluMSIDdata"))
#'
#' my_OTPICStbl <- OPTICStbl(distmat[1:50,1:50], eps_cl = 0.7)
#'
#' @export
OPTICStbl <- function(distmat, eps = 10000, minPts = 3, eps_cl = 0.5){
    opt <- dbscan::optics(  stats::as.dist(distmat),
                            eps = eps,
                            minPts = minPts,
                            search = "dist")

    res <- dbscan::extractDBSCAN(opt, eps_cl = eps_cl)

    clustmat <- data.frame(
        feature = colnames(as.matrix(distmat)),
        cluster_ID = res$cluster,
        OPTICS_order = match(seq_along(colnames(as.matrix(distmat))),
                                res$order)
    )

    return(clustmat)
}

#' Visualisation of density-based clustering of spectral similarity data
#'
#' \code{OPTICSplot()} performs density-based clustering
#' of spectral similarity data using the OPTICS algorithm like
#' \code{\link{OPTICStbl}} and creates a reachability distance plot.
#'
#' @inheritParams OPTICStbl
#'
#' @param ... Additional graphical parameters to be passed to \code{plot()}
#'
#' @return A reachability distance plot as visualisation of OPTICS clustering,
#' see code{\link[dbscan:optics]{extractDBSCAN}}.
#'
#' @details The function internally uses \code{\link[dbscan]{optics}}
#' and \code{\link[dbscan:optics]{extractDBSCAN}} from the \pkg{dbscan}
#' package.
#'
#' @seealso OPTICStbl
#'
#' @examples
#' load(file = system.file("extdata",
#'     "distmat.RData",
#'     package = "CluMSIDdata"))
#'
#' OPTICSplot(distmat[1:50,1:50], eps_cl = 0.7)
#'
#' @import dbscan
#' @import grDevices
#'
#' @importFrom stats as.dist
#' @importFrom graphics plot
#'
#' @export
OPTICSplot <- function(distmat, eps = 10000,
                                minPts = 3, eps_cl = 0.5, ...){
    opt <- dbscan::optics(  stats::as.dist(distmat),
                            eps = eps,
                            minPts = minPts,
                            search = "dist")

    res <- dbscan::extractDBSCAN(opt, eps_cl = eps_cl)

    params <- list(...)
    if(!("main" %in% names(params))) params$main <- list(NULL)

    opal <- grDevices::palette()
    grDevices::palette(c(opal, rep(c("orange", opal[-1]),10)))
    do.call(graphics::plot, append(list(x = res), params))
    grDevices::palette(opal)
}
