## ----style-knitr, eval=TRUE, echo=FALSE, results="asis"---------------------------------
BiocStyle::latex(width=90)

## ----loading----------------------------------------------------------------------------
library(iNETgrate)
set.seed(1)

## ----downloaData, eval=FALSE------------------------------------------------------------
# tcgaAML <- downloaData(dataProject="TCGA-LAML", savePath=".")

## ----load_data--------------------------------------------------------------------------
data(toyRawAml)
class(toyRawAml)
names(toyRawAml)

## ----clinical, echo=TRUE, eval=TRUE-----------------------------------------------------
clinSettings <- c("patientIDCol"="bcr_patient_barcode",
"eventCol"="vital_status",
"timeCol"="days_to_last_followup",
"riskCatCol"="acute_myeloid_leukemia_calgb_cytogenetics_risk_category",
"riskFactorCol"="cytogenetic_abnormalities",
"event"="Dead",
"riskHigh"="Poor",
"riskLow"="Favorable")
print(clinSettings)

## ----oneStep, echo=TRUE, eval=FALSE-----------------------------------------------------
# inetgrator <- iNETgrate(Data=toyRawAml, clinSettings=clinSettings,
#                         saveDir="iNETgrateOut", mus=0.6)

## ----save_results, eval=TRUE------------------------------------------------------------
resPath <- file.path(tempdir(), "iNETgrateRes") ## the result path
message(paste("Results will be saved in:", resPath, sep="\n"))
netPath <- file.path(resPath, "net")
survivalPath <- file.path(netPath, "surv")
dir.create(survivalPath, showWarnings=FALSE, recursive=TRUE)

## ----clean_data, message=FALSE, eval=TRUE-----------------------------------------------
riskCatCol <- "acute_myeloid_leukemia_calgb_cytogenetics_risk_category"
riskFactorCol <- "cytogenetic_abnormalities"
cleanedToy <- cleanAllData(genExpr=toyRawAml$genExpr,
                            genExprSampleInfo=toyRawAml$genExprSampleInfo, 
                            rawDnam=toyRawAml$rawDnam, savePath=resPath, 
                            clinical=toyRawAml$clinical, 
                            riskCatCol=riskCatCol, riskFactorCol=riskFactorCol, 
                            riskHigh="Poor", riskLow="Favorable", 
                            verbose=1)

## ----select_genes, eval=TRUE------------------------------------------------------------
data(toyCleanedAml)
cleaned <- toyCleanedAml
elected <- electGenes(genExpr=cleaned$genExpr, dnam=cleaned$dnam,
                       survival=cleaned$survival, savePath=resPath, 
                       event="Dead", locus2gene=cleaned$locus2gene, 
                       doAlLoci=FALSE, verbose=1)

## ----computing_eigenloci, eval=TRUE-----------------------------------------------------
patientLabel <- setNames(as.character(cleaned$survival$Risk1),
                         nm=rownames(cleaned$survival))
inBoth <- intersect(colnames(cleaned$dnam), names(patientLabel))
computedEloci <- computEigenloci(dnam=cleaned$dnam[ ,inBoth],
    Labels=patientLabel[inBoth],
    geNames=elected$unionGenes,
    locus2gene=cleaned$locus2gene,
    plotPath=resPath,
    Label1="Low", Label2="High",
    verbose=1)

## ----makeNetwork, eval=TRUE-------------------------------------------------------------
eigenloci <- computedEloci$eigenloci
madeNet <- makeNetwork(genExpr=cleaned$genExpr, eigenloci=eigenloci,
                            geNames=elected$unionGenes, mus=0.6, 
                            doRemoveTOM=TRUE, outPath=netPath, 
                            verbose=1)
print(madeNet$mu2modules[,1:5, drop=FALSE])               

## ----combining_eigengenes, eval=TRUE----------------------------------------------------
e1 <- computEigengenes(genExpr=cleaned$genExpr, eigenloci=eigenloci, 
                          netPath=netPath, geNames=elected$unionGenes,
                          Labels=patientLabel, Label1="Low", 
                          Label2="High", mus=c(0.6), combiningMu=NA,
                          doIgnoreNas=TRUE, survival=cleaned$survival, 
                          event="Dead", verbose=1, 
                          mu2modules=madeNet$mu2modules)

## ----analyzeSurvival, fig.width=12, fig.height=8, eval=TRUE-----------------------------
s1 <- analyzeSurvival(survival=cleaned$survival, 
                        favRisk="High", 
                        subSet="Int", mus=0.6, 
                        netPath=netPath, outPath=survivalPath, 
                        xmax1=15, xmin1=0, verbose=1)

## ----inetgrator, eval=TRUE--------------------------------------------------------------
inetgrator <- bestInetgrator(bestPvalues=s1$bestPvalues, 
                  usefuLoci=computedEloci$usefuLoci,
                  lociPigen=computedEloci$lociPigen,
                  netPath=netPath)

## ----pathway_analysis, eval=TRUE--------------------------------------------------------
selectedModules <- names(inetgrator$modules)
geneList <- list()
for(m1 in selectedModules)
    geneList[[m1]] <- names(inetgrator$modules[[m1]]$genes)
library(org.Hs.eg.db) ## Needed for human genes. 
got <- Pigengene::get.enriched.pw(geneList, idType="SYMBOL",
                 pathwayDb="KEGG", outPath=survivalPath)

## ----citation, results='asis', eval=TRUE------------------------------------------------
citation("iNETgrate")

## ----sessionInfo, results='asis', eval=TRUE---------------------------------------------
toLatex(sessionInfo())

