// RelFacts.java, created Thu Aug 18 12:08:46 2005 by salcianu
// Copyright (C) 2005 Alexandru Salcianu <salcianu@alum.mit.edu>
// Licensed under the Modified BSD Licence; see COPYING for details.
package jpaul.DataStructs;

import java.util.Set;

/**
 * <code>RelFacts</code> contains common relation factories.  For each
 * kind of relation factory, we have a corresponding static method.
 * 
 *
 * <b>Note:</b> some relation factories that used to exist as separate
 * classes are now static inner classes of this class.  They are
 * provided mostly to simplify porting old code (programmers only have
 * to change a few <code>import</code> statements).
 *
 * @author  Alexandru Salcianu - salcianu@alum.mit.edu
 * @version $Id: RelFacts.java,v 1.4 2006/01/29 16:05:28 adam_kiezun Exp $ */
public abstract class RelFacts {
    
    /** Returns a relation factory that generates
	<code>Relation</code>s interface backed by a
	<code>LinkedHashMap</code> from keys to
	<code>LinkedHashSet</code>s of values.  This relation factory
	is very good for programs that use only a few large relations.  */
    public static <K,V> RelationFactory<K,V> mapSet() {
	return mapSet(MapFacts.<K,Set<V>>hash(),
		      SetFacts.<V>hash());
    }


    /** Returns a relation factory that generates
	<code>Relation</code>s interface backed by a <code>Map</code>
	from keys to <code>Set</code>s of values.  The map is created
	by <code>mapFact</code> and the sets by <code>setFact</code>.

	The two parameters allow the programmer to finely tune the
	relations from the program.  */
    public static <K,V> RelationFactory<K,V> mapSet(MapFactory<K,Set<V>> mapFact,
						    SetFactory<V> setFact) {
	return new jpaul.DataStructs.MapSetRelationFactory<K,V>(mapFact, setFact);	
    }


    /** <code>MapSetRelationFactory</code> generates
	<code>MapSetRelation</code>s that use a specific
	<code>MapFactory</code> and a specific <code>SetFactory</code>. 

	@deprecated  As of jpaul 2.2, use {@link #mapSet()} or {@link #mapSet(MapFactory,SetFactory)} instead. */
    @Deprecated
	public static class MapSetRelationFactory<K,V> extends jpaul.DataStructs.MapSetRelationFactory<K,V> {
	/** Default constructor: creates a
	    <code>MapSetRelationFactory</code> that generates
	    relations whose representation uses
	    <code>LinkedHashMap</code>s and <code>LinkedHashSet</code>s. */
	public MapSetRelationFactory() {
	    super();
	}

	/** Creates a <code>MapSetRelationFactory</code> that
            generates relations whose representation uses maps
            generated by <code>mapFact</code> and sets generated by
            <code>setFact</code>. */
	public MapSetRelationFactory(MapFactory<K,Set<V>> mapFact, SetFactory<V> setFact) {
	    super(mapFact, setFact);
	}	
    }


    /** Returns a relation factory that generates "copy-on-write"
	(COW) relations.  A COW relation shares its representation
	(also a relation) with other COW relations, until a mutation
	occurs.  At that moment, the COW relation makes a private,
	exclusive copy of its underlying representation, and mutates
	that copy.

	<p>The internal representation of a COW representation
	maintains a "sharing" counter to identify cases when the
	representation is not shared with anyone (and hence, no
	cloning is necessary before a mutation).

	<p>Cloning a COW relation is a constant time operation.  COW
	relations are good when it is hard to determine statically
	whether a clone of a relation will be mutated: they delay the
	real cloning until the first mutation (if any).  

	@param underRelFact Relation factory for generating the
	relations used in the representation of the generated COW
	relations. */
    public static <K,V> RelationFactory<K,V> cow(RelationFactory<K,V> underRelFact) {
	return new jpaul.DataStructs.COWRelationFactory<K,V>(underRelFact);
    }

    
    /** <code>COWRelationFactory</code> generates "copy-on-write" (COW)
	relations.

	@deprecated  As of jpaul 2.2, use {@link #cow(RelationFactory)} instead. */
    @Deprecated
	public static class COWRelationFactory<K,V> extends jpaul.DataStructs.COWRelationFactory<K,V> {
	/** Creates a <code>COWRelationFactory</code>.
	    
	    @param underRelFact Relation factory for generating the
	    relations used in the representation of the generated COW
	    relations. */
	public COWRelationFactory(RelationFactory<K,V> underRelFact) {
	    super(underRelFact);
	}
    }

}
