// COWRelationFactory.java, created Fri Jul 15 09:41:02 2005 by salcianu
// Copyright (C) 2005 Alexandru Salcianu <salcianu@alum.mit.edu>
// Licensed under the Modified BSD Licence; see COPYING for details.
package jpaul.DataStructs;

import java.util.Set;
import java.util.Collection;

import jpaul.Misc.Predicate;

/**
   <code>COWRelationFactory</code> generates "copy-on-write" (COW)
   relations.  A COW relation shares its representation (also a
   relation) with other COW relations, until a mutation occurs.  At
   that moment, the COW relation makes a private, exclusive copy of
   its underlying representation, and mutates that copy. 

   <p>The internal representation of a COW representation maintains a
   "sharing" counter to identify cases when the representation is not
   shared with anyone (and hence, no cloning is necessary before a
   mutation).

   <p>Cloning a COW relation is a constant time operation.  COW
   relations are good when it is hard to determine statically whether
   a clone of a relation will be mutated: they delay the real cloning
   until the first mutation (if any).
   
 * @author  Alexandru Salcianu - salcianu@alum.mit.edu
 * @version $Id: COWRelationFactory.java,v 1.11 2006/03/21 17:37:00 adam_kiezun Exp $ */
class COWRelationFactory<K,V> extends RelationFactory<K,V> {
    
    /** Creates a <code>COWRelationFactory</code>.

	@param underRelFact Relation factory for generating the
	relations used in the representation of the COW relations
	generated by this <code>COWRelationFactory</code>. */
    public COWRelationFactory(RelationFactory<K,V> underRelFact) {
	this.underRelFact = underRelFact;        
    }

    private final RelationFactory<K,V> underRelFact;


    public Relation<K,V> create() {
	return new COWRelation<K,V>(underRelFact);
    }


    public Relation<K,V> create(Relation<K,V> rel) {
	if(rel instanceof COWRelation/*<K,V>*/) {
	    return ((COWRelation<K,V>) rel).clone();
	}
	return super.create(rel);	
    }
    

    private static class RelWithCount<K,V> {
	RelWithCount(Relation<K,V> rel) {
	    this.rel = rel;
	}
	int countOthers = 0;
	final Relation<K,V> rel;
    }


    private static class COWRelation<K,V> extends Relation<K,V> implements Cloneable {
        private static final long serialVersionUID = -3576556594053836850L;

	COWRelation(RelationFactory<K,V> underRelFact) {
	    this.underRelFact = underRelFact;
	    this.underRel = new RelWithCount<K,V>(underRelFact.create());
	}

	private final RelationFactory<K,V> underRelFact;
	private RelWithCount<K,V> underRel;

	/* Synchronization: please read the synchronization discussion
	   from COWSetFactory.  It applies here verbatim.  */

	public boolean add(K key, V value) {
	    if(shared()) {
		if(contains(key, value)) return false;
		detach();
	    }
	    return underRel.rel.add(key, value);
	} 
	
	public boolean addAll(K key, Collection<V> values) {
	    if(shared()) {
		if(containsAll(key, values)) return false;
		detach();		
	    }
	    return underRel.rel.addAll(key, values);
	}

	public boolean addAll2(K key, Collection<? extends V> values) {
	    if(shared()) {
		if(containsAll(key, values)) return false;
		detach();		
	    }
	    return underRel.rel.addAll2(key, values);
	}

	public void clear() {
	    if(shared()) {
		if(isEmpty()) return;
		detach();
	    }
	    underRel.rel.clear();
	}

	public boolean remove(K key, V value) {
	    if(shared()) {
		if(!contains(key, value)) return false;
		detach();
	    }
	    return underRel.rel.remove(key, value);	    
	}

	public boolean removeAll(K key, Collection<V> values) {
	    if(shared()) {
		detach();
	    }
	    return underRel.rel.removeAll(key, values);
	}

	public boolean removeKey(K key) {
	    if(shared()) {
		if(!containsKey(key)) return false;
		detach();
	    }
	    return underRel.rel.removeKey(key);
	}

	public boolean removeKeys(Predicate<K> predicate) {
	    if(shared()) {
		detach();
	    }
	    return underRel.rel.removeKeys(predicate);
	}

	public boolean removeValues(Predicate<V> predicate) {
	    if(shared()) {
		detach();
	    }
	    return underRel.rel.removeValues(predicate);
	}


	public boolean contains(K key, V value) { return underRel.rel.contains(key, value); }
	public boolean containsKey(K key)       { return underRel.rel.containsKey(key); }
	public boolean isEmpty()                { return underRel.rel.isEmpty(); }
	protected Set<V> _getValues(K key)      { return underRel.rel._getValues(key); }
	public    Set<K> keys()                 { return underRel.rel.keys(); }
	public    Iterable<V> values()          { return underRel.rel.values(); }

	public boolean union(Relation<K,V> rel) {
	    if(rel instanceof COWRelation/*<K,V>*/) {
		COWRelation<K,V> cowRel2 = (COWRelation<K,V>) rel;
		if(this.underRel == cowRel2.underRel)
		    return false;
	    }
	    if(shared()) {
		detach();
	    }
	    return this.underRel.rel.union(rel);
	}

	public boolean equals(Object o) {
	    if(o == null) return false;
	    if(o == this) return true;
	    if(o instanceof COWRelation/*<K,V>*/) {
		@SuppressWarnings("unchecked")
		COWRelation<K,V> cowRel2 = (COWRelation<K,V>) o;
		if(this.underRel == cowRel2.underRel) return true;
		return this.underRel.rel.equals(cowRel2.underRel.rel);
	    }
	    return this.underRel.rel.equals(o);
	}

	public COWRelation<K,V> clone() {
	    synchronized(underRel) {
		// do the increment before creating a new reference
		// (via super.clone() below) such that the sharing
		// count is always conservative.
		underRel.countOthers++;
	    }
	    COWRelation<K,V> copy = (COWRelation<K,V>) super.clone();
	    return copy;
	}

	protected void finalize() {
	    try {
		try {
		    synchronized(underRel) {
			underRel.countOthers--;
		    }
		}
		finally {
		    super.finalize();
		}
	    }
	    catch(Throwable tex) {
		throw new Error(tex);
	    }
	}

	private final boolean shared() { return underRel.countOthers > 0; }

	private final void detach() {
	    RelWithCount<K,V> oldUnderRel = this.underRel;
	    this.underRel = new RelWithCount<K,V>(underRelFact.create(oldUnderRel.rel));
	    // do the decrement HERE, after the real detach, to make
	    // sure that the sharing count is conservative at any point.
	    oldUnderRel.countOthers--;
	}

	public String toString() {
	    String stdStr = underRel.rel.toString();
	    return 
		shared() ?
		("(shared: " + (underRel.countOthers+1) + ") " + stdStr) :
		stdStr;
	}

    }

}
