% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/metascope_blast.R
\name{metascope_blast}
\alias{metascope_blast}
\title{Blast reads from MetaScope aligned files}
\usage{
metascope_blast(
  metascope_id_path,
  bam_file_path = list.files(tmp_dir, ".updated.bam$", full.names = TRUE)[1],
  tmp_dir,
  out_dir,
  sample_name,
  fasta_dir = NULL,
  num_results = 10,
  num_reads = 100,
  hit_list = 10,
  num_threads = 1,
  db_path,
  quiet = FALSE,
  db = NULL,
  accession_path = NULL
)
}
\arguments{
\item{metascope_id_path}{Character string; path to a csv file output by
`metascope_id()`.}

\item{bam_file_path}{Character string; full path to bam file for the same
sample processed by `metascope_id`. Note that the `filter_bam` function
must have output a bam file, and not a .csv.gz file. See
`?filter_bam_bowtie` for more details. Defaults to
\code{list.files(file_temp, ".updated.bam$")[1]}.}

\item{tmp_dir}{Character string, a temporary directory in which to host
files.}

\item{out_dir}{Character string, path to output directory.}

\item{sample_name}{Character string, sample name for output files.}

\item{fasta_dir}{Directory where fasta files for blast will be stored.}

\item{num_results}{Integer, the maximum number of taxa from the metascope_id
output to check reads. Takes the top n taxa, i.e. those with largest
abundance. Default is 10.}

\item{num_reads}{Integer, the maximum number of reads to blast per microbe.
If the true number of reads assigned to a given taxon is smaller, then the
smaller number will be chosen. Default is 100. Too many reads will involve
more processing time.}

\item{hit_list}{Integer, number of blast hit results to keep. Default is 10.}

\item{num_threads}{Integer, number of threads if running in parallel
(recommended). Default is 1.}

\item{db_path}{Character string. The database file to be searched (including
basename, but without file extension). For example, if the nt database is
in the nt folder, this would be /filepath/nt/nt assuming that the database
files have the nt basename. Check this path if you get an error message
stating "No alias or index file found".}

\item{quiet}{Logical, whether to print out more informative messages. Default
is FALSE.}

\item{db}{Currently accepts one of \code{c("ncbi", "silva", "other")} Default
is \code{"ncbi"}, appropriate for samples aligned against indices compiled
from NCBI whole genome databases. Alternatively, usage of an alternate
database (like Greengenes2) should be specified with \code{"other"}.}

\item{accession_path}{(character) Filepath to NCBI accessions SQL
database. See \code{taxonomzr::prepareDatabase()}.}
}
\value{
This function writes an updated csv file with metrics.
}
\description{
This function allows the user to check a subset of identified reads against
NCBI BLAST and the nucleotide database to confirm or contradict results
provided by MetaScope. It aligns the top `metascope_id()` results to NCBI
BLAST database. It REQUIRES that command-line BLAST and a separate nucleotide
database have already been installed on the host machine. It returns a csv
file updated with BLAST result metrics.
}
\details{
This function assumes that you used the NCBI nucleotide database to process
samples, with a download date of 2021 or later. This is necessary for
compatibility with the bam file headers.

This is highly computationally intensive and should be ran with multiple
cores, submitted as a multi-threaded computing job if possible.

Note, if best_hit_strain is FALSE, then no strain was observed more often
among the BLAST results.
}
\examples{
\dontrun{
### Create temporary directory
file_temp <- tempfile()
dir.create(file_temp)

bamPath <- system.file("extdata", "bowtie_target.bam",
                       package = "MetaScope")
file.copy(bamPath, file_temp)

metascope_id(file.path(file_temp, "bowtie_target.bam"), aligner = "bowtie2",
             input_type = "bam", out_dir = file_temp, num_species_plot = 0,
             update_bam = TRUE)

## Run metascope blast
### Get export name and metascope id results
out_base <- bamPath |> basename() |> tools::file_path_sans_ext() |>
  tools::file_path_sans_ext()
metascope_id_path <- file.path(file_temp, paste0(out_base, ".metascope_id.csv"))

# NOTE: change db_path to the location where your BLAST database is stored!
db <- "/restricted/projectnb/pathoscope/data/blastdb/nt/nt"

tmp_accession <- system.file("extdata", "example_accessions.sql", package = "MetaScope"

metascope_blast(metascope_id_path,
                bam_file_path = file.path(file_temp, "bowtie_target.bam"),
                tmp_dir = file_temp,
                out_dir = file_temp,
                sample_name = out_base,
                db_path = db,
                num_results = 10,
                num_reads = 5,
                hit_list = 10,
                num_threads = 3,
                db = "ncbi",
                quiet = FALSE,
                fasta_dir = NULL,
                accession_path = tmp_accession)

## Remove temporary directory
unlink(file_temp, recursive = TRUE)
}

}
