% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/knnClassify.R
\name{knnClassify}
\alias{knnClassify}
\alias{knnClassify,data.frame-method}
\alias{knnClassify,ddpcrWell-method}
\alias{knnClassify,ddpcrPlate-method}
\title{Use the k-nearest neighbour algorithm to classify the wells in
a \code{ddpcrWell} or \code{ddpcrPlate} object, or in a data frame.}
\usage{
knnClassify(droplets, trainData, cl, k, prob = 0, ...)

\S4method{knnClassify}{data.frame}(droplets, trainData, cl, k, prob = 0, fullTable = TRUE)

\S4method{knnClassify}{ddpcrWell}(droplets, trainData, cl, k, prob = 0)

\S4method{knnClassify}{ddpcrPlate}(droplets, trainData, cl, k, prob = 0)
}
\arguments{
\item{droplets}{A \code{\link{ddpcrWell}} or \code{\link{ddpcrPlate}}
object, or a data frame with columns \code{Ch1.Amplitude} and
\code{Ch2.Amplitude}.}

\item{trainData}{A data frame of training data with columns
\code{Ch1.Amplitude} and \code{Ch2.Amplitude}.}

\item{cl}{A vector of classes corresponding to \code{trainData}.}

\item{k}{The number of nearest neighbours to use in the algorithm.}

\item{prob}{The minimal proportion of votes for the winning class needed to
assert that a droplet belongs to the class. This figure should be a float
between 0 and 1. For example, if 0.6 then at least 60% of a droplet's
k-nearest neighbours need to be of one class, otherwise it is classified as
"Rain". Defaults to 0, i.e. we do not use "Rain".}

\item{...}{Other options depending on the type of \code{droplets}.}

\item{fullTable}{If \code{TRUE}, returns a full data frame of droplets and
their classification; if \code{FALSE}, simply returns the classified vector.
Defaults to \code{TRUE}.}
}
\value{
An object with the new classification.

If \code{droplets} is a data frame, return data frame or factor
(depending on the value of \code{fullTable}) of the droplet classification
under the k-NN algorithm.
}
\description{
If \code{droplets} is a data frame, the droplets are classified
using the k-nearest neighbour algorithm against a training data set.

If \code{droplets} is a \code{ddpcrWell} object, the droplets
in the well are classified and returned in another \code{ddpcrWell} object.

If \code{droplets} is a \code{ddpcrPlate} object, the
wells are combined and classified together, with the resulting
classification assigned to the \code{ddpcrPlate} object.
}
\examples{
### Use the KRASdata dataset for all of these examples.

## Use k-means clustering to classify one sample. Use this as training
## data for the K-Nearest Neighbour algorithm.
trainingData <- KRASdata[["E03"]]
trainingData <- kmeansClassify(trainingData)$data

## Classify a dataframe using k-NN with k = 1 and the above training data.
aWell <- knnClassify(
           KRASdata[["F03"]],
           trainData=trainingData[, c("Ch1.Amplitude", "Ch2.Amplitude")],
           cl=trainingData$class,
           k=1)
dropletPlot(aWell, cMethod="class")  # visualising the classification

## We can change k to a larger number, here with a ddpcrWell object.
aWell <- ddpcrWell(well=KRASdata[["E03"]])
aWell <- knnClassify(
           aWell,
           trainData=trainingData[, c("Ch1.Amplitude", "Ch2.Amplitude")],
           cl=trainingData$class,
           k=3)
dropletPlot(aWell, cMethod="knn")  # visualising the classification

## Changing the 'prob' parameter means that droplets with less than 'prob'
## of the votes will not be classified. We do this for a ddpcrPlate
## object.
krasPlate <- ddpcrPlate(wells=KRASdata[c("E03", "H03", "C04", "F04")])
krasPlate <- knnClassify(
               krasPlate,
               trainData=trainingData[, c("Ch1.Amplitude", "Ch2.Amplitude")],
               cl=trainingData$class,
               k=3,
           prob=0.6)
dropletPlot(krasPlate, cMethod="knn")  # visualising the classification

}
\seealso{
This method uses the \code{\link[class]{knn}} function.

To manually set and retrieve classifications, use the
\code{\link{wellClassification}}, \code{\link{plateClassification}} and
\code{\link{plateClassificationMethod}} methods.

\code{kmeansClassify} provides a wrapper for the k-means clustering
algorithm.
}
\author{
Anthony Chiu, \email{anthony.chiu@cruk.manchester.ac.uk}
}
