% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/scoreSingleSamples.R
\name{scoreSingleSamples}
\alias{scoreSingleSamples}
\title{Generates single sample gene set scores across a datasets by many methods}
\usage{
scoreSingleSamples(
  gdb,
  y,
  methods = "ewm",
  as.matrix = FALSE,
  drop.sd = 1e-04,
  drop.unconformed = FALSE,
  verbose = FALSE,
  recenter = FALSE,
  rescale = FALSE,
  ...,
  as.dt = FALSE
)
}
\arguments{
\item{gdb}{A GeneSetDb}

\item{y}{An expression matrix to score genesets against}

\item{methods}{A character vector that enumerates the scoring methods you
want to run over the samples. Please reference the "Single Sample Scoring
Methods" section for more information.}

\item{as.matrix}{Return results as a list of matrices instead of a melted
data.frame? Defaults to \code{FALSE}.}

\item{drop.sd}{Genes with a standard deviation across columns in \code{y}
that is less than this value will be dropped.}

\item{drop.unconformed}{When \code{TRUE}, genes in \code{y} that are not found in
\code{gdb} are removed from the expression container. You may want to set this
to \code{TRUE} when \code{y} is very large until better sparse matrix support is
injected. This will change the scores for gsva and ssGSEA, though.
Default is \code{FALSE}.}

\item{verbose}{make some noise? Defaults to \code{FALSE}.}

\item{recenter, rescale}{If \code{TRUE}, the scores computed by each method
are centered and scaled using the \code{scale} function. These variables
correspond to the \code{center} and \code{scale} parameters in the
\code{scale} function. Defaults to \code{FALSE}.}

\item{...}{these parameters are passed down into the the individual single
sample scoring funcitons to customize them further.}

\item{as.dt}{If \code{FALSE} (default), the data.frame like thing that
this funciton returns will be set to a data.frame. Set this to \code{TRUE}
to keep this object as a \code{data.table}}
}
\value{
A long data.frame with sample_id,method,score values per row. If
\code{as.matrix=TRUE}, a matrix with as many rows as \code{geneSets(gdb)}
and as many columns as \code{ncol(x)}
}
\description{
It is common to assess the activity of a gene set in a given sample. There
are many ways to do that, and this method is analogous to the
\code{\link[=seas]{seas()}} function in that it enables the user to run a multitude of
single-sample-gene-set-scoring algorithms over a target expression matrix
using a \code{\link[=GeneSetDb]{GeneSetDb()}} object.
}
\details{
Please refer to the "Generating Single Sample Gene Set Scores" of the
sparrow vignette for further exposition.
}
\section{Single Sample Scoring Methods}{

The following \code{methods} are currenly provided.
\itemize{
\item \code{"ewm"}: The \code{\link[=eigenWeightedMean]{eigenWeightedMean()}} calculates the fraction each gene
contributes to a pre-specified principal component. These contributions
act as weights over each gene, which are then used in a simple weighted
mean calculation over all the genes in the geneset per sample. This is
similar, in spirit, to the svd/gsdecon method (ie. \verb{method = "gsd"``) You can use this method to perform an "eigenweighted" zscore by setting }unscale\code{and}uncenter\code{to}FALSE\code{. }"ewz"\verb{: with }unscale\code{and}uncenter\verb{set to}FALSE`.
\item \code{"gsd"}: This method was first introduced by Jason Hackney in
\href{https://doi.org/10.1038/ng.3520}{doi:10.1038/ng.3520}. Please refer to
the \code{\link[=gsdScore]{gsdScore()}} function for more information.
\item \code{"ssgsea"}: Using ssGSEA as implemented in the GSVA package.
\item \code{"zscore"}: The features in the expression matrix are rowwise z transformed.
The gene set level score is then calculated by adding up the zscores for
the genes in the gene set, then dividing that number by either the the
size (or its sqaure root (default)) of the gene set.
\item \code{"mean"}: Simply take the mean of the values from the expression matrix
that are in the gene set. Right or wrong, sometimes you just want the mean
without transforming the data.
\item \code{"gsva"}: The gsva method of GSVA package.
\item \code{"plage"}: Using "plage" as implemented in the GSVA package
}
}

\examples{
gdb <- exampleGeneSetDb()
vm <- exampleExpressionSet()
scores <- scoreSingleSamples(
  gdb, vm, methods = c("ewm", "gsva", "zscore"),
  center = TRUE, scale = TRUE, ssgsea.norm = TRUE, as.dt = TRUE)

sw <- data.table::dcast(scores, name + sample_id ~ method, value.var='score')

\donttest{
corplot(
  sw[, c("ewm", "gsva", "zscore")],
  title = "Single Sample Score Comparison")
}

zs <- scoreSingleSamples(
  gdb, vm, methods = c('ewm', 'ewz', 'zscore'), summary = "mean",
  center = TRUE, scale = TRUE, uncenter = FALSE, unscale = FALSE,
  as.dt = TRUE)
zw <- data.table::dcast(zs, name + sample_id ~ method, value.var='score')

\donttest{
  corplot(zw[, c("ewm", "ewz", "zscore")], title = "EW zscores")
}
}
