\name{denspr}
\alias{denspr}
\title{Density Estimation}
\description{
  Estimates the density of a vector of observations by a Poisson regression
  fit to histogram counts.
}
\usage{
  denspr(x, n.interval = NULL, df = 5, knots.mode = TRUE, 
      type.nclass = c("wand", "scott", "FD"), addx=FALSE)
}
\arguments{
  \item{x}{a numeric vector containing the observations for which the density
     should be estimated.}
  \item{n.interval}{an integer specifying the number of cells for the histogram.
     If \code{NULL}, \code{n.interval} is estimated by the method specified by
     \code{type.nclass}.}
  \item{df}{integer specifying the degrees of freedom of the natural cubic spline
     used in the Poisson regression fit.}
  \item{knots.mode}{if \code{TRUE} the \code{df} - 1 knots are centered around the
     mode and not the median of the density, where the mode is estimated by the 
     midpoint of the cell of the histogram that contains the largest number of 
     observations. If \code{FALSE}, the default knots are used in the function \code{ns}.
     Thus, if \code{FALSE} the basis matrix will be generated by \code{ns(x, df = 5)}.}
  \item{type.nclass}{character string specifying the procedure used to compute the
     number of cells of the histogram. Ignored if \code{n.interval} is specified.
     By default, the method of Wand (1994) with 
     \code{level = 1} (see the help page of \code{dpih} in the package \pkg{KernSmooth}) is used.
     For the other choices, see \code{\link{nclass.scott}}.}
  \item{addx}{should \code{x} be added to the output? Necessary when the estimated density
     should be plotted by \code{plot(out)} or \code{lines(out)}, where \code{out} is
     the output of \code{denspr}.}
}

\value{
  An object of class \code{denspr} consisting of 
  \item{y}{a numeric vector of the same length as \code{x} containing the estimated density
     for each of the observations}
  \item{center}{a numeric vector specifying the midpoints of the cells of the histogram}
  \item{counts}{a numeric vector of the same length as \code{center} composed of the number
     of observations of the corresponding cells}
  \item{x.mode}{the estimated mode}
  \item{ns.out}{the output of \code{ns}}
  \item{type}{the method used to estimate the numbers of cells}
  \item{x}{the input vector \code{x} if \code{addx = TRUE}; otherwise, \code{NULL}.}
}

\references{
  Efron, B., and Tibshirani, R. (1996). Using specially designed exponential 
  families for density estimation. \emph{Annals of Statistics}, 24, 2431--2461.

  Wand, M.P. (1997). Data-based choice of histogram bin width. 
  \emph{American Statistician}, 51, 59--64. 
}

\author{Holger Schwender,\email{holger.schw@gmx.de}}

\seealso{\code{\link{cat.ebam}}}

\examples{\dontrun{
# Generating some random data.
x <- rnorm(10000)
out <- denspr(x, addx=TRUE)
plot(out)

# Or for an asymmetric density.
x <- rchisq(10000, 2)
out <- denspr(x, df=3, addx=TRUE)
plot(out)
}}

\keyword{smooth}
