% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/se_quickRnaQc.R
\name{quickRnaQc.se}
\alias{quickRnaQc.se}
\alias{computeRnaQcMetricsWithAltExps}
\alias{formatComputeRnaQcMetricsResult}
\title{Quick quality control for RNA data in a SummarizedExperiment}
\usage{
quickRnaQc.se(
  x,
  subsets,
  num.threads = 1,
  block = NULL,
  more.suggest.args = list(),
  altexp.proportions = NULL,
  assay.type = "counts",
  output.prefix = NULL,
  meta.name = "qc",
  flatten = TRUE
)

computeRnaQcMetricsWithAltExps(
  x,
  subsets,
  altexp.proportions,
  num.threads = 1,
  assay.type = "counts"
)

formatComputeRnaQcMetricsResult(compute.res, flatten = TRUE)
}
\arguments{
\item{x}{A \link[SummarizedExperiment]{SummarizedExperiment} object or one of its subclasses.
Rows correspond to genes and columns correspond to cells.}

\item{subsets}{List of subsets of control genes, see \code{?\link{computeRnaQcMetrics}} for more details.}

\item{num.threads}{Number of threads, to pass to \code{\link{computeRnaQcMetrics}}.}

\item{block}{Block assignment for each cell, to pass to \code{\link{suggestRnaQcThresholds}} and \code{\link{filterRnaQcMetrics}}.}

\item{more.suggest.args}{Named list of additional arguments to pass to \code{\link{suggestRnaQcThresholds}}.}

\item{altexp.proportions}{Alternative experiments for which to compute QC metrics.
This is typically used to refer to alternative experiments holding spike-in data.
For each alternative experiment, the proportion is defined as \eqn{X/(X+Y)} where \eqn{X} is the alternative experiment's total and \eqn{Y} is the RNA total.
These proportions will be used for filtering in the same manner as the proportions computed from \code{subsets}.

More specifically, \code{altexp.proportions} should be an unnamed integer or character vector containing the names/indices of the alternative experiments of interest.
The assay to use from each alternative experiment is determined by \code{assay.type}.

Alternatively, \code{altexp.proportions} may be a named integer or character vector.
Each name specifies an alternative experiment while each value is the index/name of the assay to use from that experiment.

Only relevant if \code{x} is a \link[SingleCellExperiment]{SingleCellExperiment}.}

\item{assay.type}{Integer or string specifying the assay of \code{x} containing the RNA count matrix.}

\item{output.prefix}{String containing a prefix to add to the names of the \code{link[SummarizedExperiment]{colData}} columns containing the output statistics.}

\item{meta.name}{String containing the name of the \code{\link[S4Vectors]{metadata}} entry containing the additional outputs such as the filtering thresholds.
If \code{NULL}, additional outputs are not reported.}

\item{flatten}{Logical scalar indicating whether to flatten the subset proportions into separate columns of the \code{link[SummarizedExperiment]{colData}}.
If \code{FALSE}, the subset proportions are stored in a nested \link[S4Vectors]{DataFrame}.}

\item{compute.res}{List returned by \code{\link{computeRnaQcMetrics}}.}
}
\value{
For \code{quickRnaQc.se}, \code{x} is returned with additional columns added to its \code{\link[SummarizedExperiment]{colData}}.
Each column contains per-cell values for one of the QC metrics, see \code{\link{computeRnaQcMetrics}} for details.
The suggested thresholds are stored as a list in \code{\link[S4Vectors]{metadata}}.
The \code{colData} also contains a \code{keep} column, specifying which cells are to be retained.
If \code{altexp.proportions} is provided, QC metrics are added to the \code{colData} of the specified alternative experiments in the output object.

For \code{computeRnaQcMetricsWithAltExps}, a list is returned containing:
\itemize{
\item \code{main}, the result of calling \code{\link{computeRnaQcMetrics}} on the RNA count matrix in \code{x}.
The proportion of counts in each alternative experiment is added to the \code{subsets}.
\item \code{altexp}, a named list of length equal to \code{altexp.proportions}.
Each inner list is the result of calling \code{computeRnaQcMetrics} on the RNA count matrix of the corresponding alternative experiment of \code{x}.
}

For \code{formatComputeRnaQcMetricsResult}, a \link[S4Vectors]{DataFrame} is returned containing the per-cell QC metrics.
}
\description{
Quickly compute quality control (QC) metrics, thresholds and filters from RNA data in a \link[SummarizedExperiment]{SummarizedExperiment}.
This calls \code{\link{computeRnaQcMetrics}} on an assay in a SummarizedExperiment,
followed by \code{\link{suggestRnaQcThresholds}} and \code{\link{filterRnaQcMetrics}} to identify high-quality cells.
}
\examples{
library(SingleCellExperiment)
sce <- getTestRnaData.se()
sce <- quickRnaQc.se(sce, subsets=list(mito=grepl("^mt", rownames(sce))))
colData(sce)[,c("sum", "detected", "mito.proportion")]
metadata(sce)$qc$thresholds
summary(sce$keep)

# Computing spike-in proportions, if available.
sce <- getTestRnaData.se()
sce <- quickRnaQc.se(
   sce,
   subsets=list(mito=grepl("^mt", rownames(sce))),
   altexp.proportions="ERCC"
)
colData(sce)[,c("sum", "detected", "mito.proportion", "ERCC.proportion")]
colData(altExp(sce, "ERCC"))[,c("sum", "detected")]

}
\author{
Aaron Lun
}
