# Server -----------------------------------------------------------------------
app_server <- function(input, output, session) {
    # Setup reactivity ---------------------------------------------------------
    plots_visible <- reactiveVal(FALSE)
    plots_visible_2 <- reactiveVal(FALSE)
    plots_visible_3 <- reactiveVal(FALSE)

    continue <- reactiveVal(FALSE)

    shinyjs::disable("filter_btn")
    shinyjs::disable("continue_var")
    shinyjs::disable("kmeans_btn")

    # Upload settings
    options(shiny.maxRequestSize = 500 * 1024^2)

    # End of setting up  -------------------------------------------------------
    # Process input ------------------------------------------------------------
    ## Check input -------------------------------------------------------------
    h5_file <- reactive({
        req(input$upload) # Ensure input$upload exists

        # Open the HDF5 file
        h5_file <- H5Fopen(input$upload$datapath)

        # Check existence of important slots in input
        check_result <- checkH5(h5_file)

        # If input is valid upload gets deactivated and buttons are activated
        if (check_result$valid) {
            shinyjs::enable("filter_btn") # Enable button if valid
            shinyjs::disable("upload") # Deactivate input if valid
            output$status <- renderText("File uploaded successfully!")
            return(h5_file)
        } else {
            showNotification(
                paste(
                    "Error: The following paths are missing in the H5 file:",
                    paste(check_result$missing, collapse = ", ")
                ),
                type = "error",
                duration = NULL
            )
            return(NULL) # Return NULL if invalid
        }
    })

    # Display conditional panels if file is uploaded successfully
    output$file_ready <- reactive({
        !is.null(h5_file())
    })
    outputOptions(output, "file_ready", suspendWhenHidden = FALSE)

    # Panel analyses------------------------------------------------------------
    sce <- reactive({
        req(input$upload$datapath)
        sce_amp <- h5ToSce(input$upload$datapath)$sce_amp
        norm <- normalizeReadCounts(sce = sce_amp)
        return(norm)
    })

    se.var <- reactive({
        req(input$upload$datapath)
        return(h5ToSce(input$upload$datapath)$se_var)
    })

    # Variant analyses ---------------------------------------------------------
    ## Variant filtering -------------------------------------------------------
    # Filter variants after filter button hit
    rv <- reactiveValues(sce_filtered = NULL)
    observeEvent(input$filter_btn, {
        plots_visible(TRUE)
        current_sce <- sce()
        se.var <- se.var()
        filteres <- filterVariants(
            depth.threshold = 10,
            genotype.quality.threshold = 30,
            vaf.ref = 5,
            vaf.het = 35,
            vaf.hom = 95,
            min.cell = 50,
            min.mut.cell = 1,
            se.var = se.var,
            sce = current_sce,
            shiny = TRUE
        )

        indices_to_keep <- match(filteres$cells.keep, colData(current_sce)[[1]],
            nomatch = 0
        )
        if (length(indices_to_keep) > 0) {
            se.f <- SummarizedExperiment(
                assays = list(
                    VAF = t(filteres$vaf.matrix.filtered),
                    Genotype = t(filteres$genotype.matrix.filtered),
                    Genoqual = t(filteres$genoqual.matrix.filtered)
                ),
                rowData = filteres$variant.ids.filtered,
                colData = filteres$cells.keep
            )
            sce_filtered <- current_sce[, indices_to_keep, drop = FALSE]
            SingleCellExperiment::altExp(sce_filtered, "variants") <- se.f
            sce_filtered <- annotateVariants(sce = sce_filtered, shiny = TRUE)

            # Update the reactive value
            rv$sce_filtered <- sce_filtered
        } else {
            showModal(modalDialog(
                title = "Warning",
                "No cells passed the filtering criteria."
            ))
        }

        # Expose the visibility state
        output$plots_visible <- reactive({
            plots_visible()
        })
        outputOptions(output, "plots_visible", suspendWhenHidden = FALSE)

        ## Variant panel plots -------------------------------------------------
        ## Plot: No of variants ------------------------------------------------
        output$var_plot1 <- renderPlot({
            req(plots_visible())
            req(rv$sce_filtered)

            plot(0, type = "n", axes = FALSE, ann = FALSE)
            mtext(dim(se.var())[1], side = 3, line = -2, cex = 3, 
                col = "forestgreen")
            mtext("Number of variants total", side = 3, line = -4, cex = 1.5)

            # # Print ean mapped reads per cell
            mtext(dim(altExp(rv$sce_filtered))[1],
                side = 1, line = -4, cex = 3,
                col = "dodgerblue"
            )
            mtext("Number of variants filtered", side = 1, line = -2, cex = 1.5)

            # Draw box
            box(which = "outer", lty = "solid", col = "grey")
        })

        ## Plot: No of cells ---------------------------------------------------
        output$var_plot2 <- renderPlot({
            req(plots_visible())

            # Print number of cells
            plot(0, type = "n", axes = FALSE, ann = FALSE)
            mtext(dim(sce())[2], side = 3, line = -2, cex = 3, 
                col = "forestgreen")
            mtext("Number of cells total", side = 3, line = -4, cex = 1.5)

            # Print ean mapped reads per cell
            mtext(dim(altExp(rv$sce_filtered))[2],
                side = 1, line = -4, cex = 3,
                col = "dodgerblue"
            )
            mtext("Number of cells filtered", side = 1, line = -2, cex = 1.5)

            # Draw box
            box(which = "outer", lty = "solid", col = "grey")
        })

        ## Heatmap: VAF I ------------------------------------------------------
        output$var_plot3 <- renderPlot({
            req(rv$sce_filtered)
            plotVariantHeatmap(rv$sce_filtered)
        })

        output$legend <- renderPlot({
            # prep legend
            labels <- c("Hom", "Het", "WT", "Missing")
            colors <- c(
                `Hom` = "#D44292",
                `Het` = "#F6A97A",
                `WT` = "#414487FF",
                `Missing` = "#868686FF"
            )

            # Create the legend
            lgd <- Legend(
                labels = labels,
                title = "Genotype",
                legend_gp = gpar(fill = colors)
            )
            draw(lgd)
        })

        ## Violin: GQ ----------------------------------------------------------
        output$var_plot5 <- renderPlot({
            plotGenotypequalityPerGenotype(sce_filtered) +
                theme(
                    title = element_text(size = 20),
                    text = element_text(size = 16)
                )
        })

        # Variant tables -------------------------------------------------------
        ## DT: overview filtered variants --------------------------------------
        output$data_table_var <- renderDataTable({
            sample.name <- sce_filtered@metadata[["sample_name"]]
            file.out <- paste0("scafari_variants_", sample.name)
            rowData(altExp(sce_filtered)) %>%
                as.data.frame() %>%
                # dplyr::select(-any_of("id")) %>%
                replace(is.na(.), "-") %>%
                # rownames_to_column(var = "Variant") %>%
                tidyr::separate(id, c("Chromosome", "Position", "Alt", "Ref"),
                    sep = ":|/", remove = FALSE
                ) %>%
                dplyr::mutate(Protein = ifelse(str_detect(Protein, "\\?"),
                    Gene, Protein
                )) %>%
                dplyr::relocate(Gene, .before = Chromosome) %>%
                arrange(Gene) %>%
                datatable(.,
                    extensions = "Buttons",
                    options = list(
                        pageLength = 25, width = "95%",
                        dom = "Bfrtip",
                        buttons = list(
                            list(extend = "csv", filename = file.out),
                            list(extend = "excel", filename = file.out),
                            list(extend = "pdf", filename = file.out),
                            list(extend = "copy", filename = file.out)
                        )
                    ),
                    rownames = FALSE
                )
        })


        ## DT: select variants of interest -------------------------------------
        output$data_table_var2 <- renderDataTable({
            sort(paste0(
                rowData(altExp(sce_filtered))$Gene, ":",
                rowData(altExp(sce_filtered))$id
            )) %>%
                as.data.frame() %>%
                datatable(.,
                    options = list(
                        pageLength = 25, width = "95%",
                        scrollY = "600px",
                        paging = FALSE,
                        info = FALSE,
                        dom = "tf"
                    ), rownames = FALSE
                )
        })

        # Text: selected variants of interest ----------------------------------
        output$selected_rows <- renderPrint({
            cat((sort(paste0(
                rowData(altExp(sce_filtered))$Gene, ":",
                rowData(altExp(sce_filtered))$id
            )) %>%
                as.data.frame() %>%
                mutate(id = paste0(Gene, ":", rownames(.))) %>%
                dplyr::select(id) %>%
                arrange(id))[input$data_table_var2_rows_selected, ])
        })

        # Plots explore panel --------------------------------------------------
        ## Heatmap: VAF II -----------------------------------------------------
        output$hm_1 <- renderPlot({
            req(rv$sce_filtered)
            plotVariantHeatmap(rv$sce_filtered)
        })

        # Reactive variable to store user selections
        selected_variants <- reactiveVal(NULL)

        # Render the hm only when submit_var clicked and selected_variants !NULL
        # Update selected variants only when the button is clicked
        observeEvent(input$submit_var, {
            shinyjs::enable("continue_var")

            selected_variants(input$data_table_var2_rows_selected) 
        })

        observeEvent(input$submit_var, {
            # Actualize heatmap
            output$hm_1 <- renderPlot({
                req(selected_variants())
                req(rv$sce_filtered) 
                sce_filtered <- rv$sce_filtered
                output$hm_1 <- renderPlot({
                    variant.ids.filtered.gene <- 
                        paste0(rowData(altExp(sce_filtered))$Gene, ":", 
                            rowData(altExp(sce_filtered))$id)
                    selected_variants_id <- 
                        order(variant.ids.filtered.gene)[selected_variants()]

                    # Load matrices
                    vaf_matrix_filtered <- as.data.frame(t(assay(altExp(
                        sce_filtered, "variants"), "VAF")))
                    colnames(vaf_matrix_filtered) <- 
                        paste0(rowData(altExp(sce_filtered, "variants"))$Gene, 
                            ":", rowData(altExp(sce_filtered, "variants"))$id)
                    genotype.matrix.filtered <- 
                        as.data.frame(t(assay(altExp(sce_filtered), 
                                            "Genotype")))
                    colnames(genotype.matrix.filtered) <- 
                        paste0(rowData(altExp(sce_filtered))$Gene, ":", 
                            rowData(altExp(sce_filtered))$id)
                    row_data <- rowData(altExp(sce_filtered, "variants"))
                    names <- paste0(row_data$Gene, ":", 
                                    as.character(row_data$id))

                    vaf.matrix.filtered.hm <- 
                        vaf_matrix_filtered[, selected_variants_id] 

                    column_ha <- HeatmapAnnotation(
                        chr = factor(str_extract(colnames(
                            vaf.matrix.filtered.hm), "chr(\\d|X|Y)+"),
                            levels = chromosomes
                        ),
                        col = list(chr = chr_palette)
                    )

                    collect <- data.frame(row.names = "")

                    # GT matrix annotation (customize as needed)
                    df <- do.call(rbind, lapply(genotype.matrix.filtered, 
                                                function(x) {
                        length(x) <- 4
                        return(x)
                    }))

                    # Transform numerical genotype to WT, Het, Ho,, Missing df
                    gt.anno <- data.frame(
                        WT = integer(), Het = integer(),
                        Hom = integer(), Missing = integer()
                    )
                    for (col in seq(1, ncol(genotype.matrix.filtered))) {
                        wt <- sum(genotype.matrix.filtered[, col] == 0)
                        het <- sum(genotype.matrix.filtered[, col] == 1)
                        hom <- sum(genotype.matrix.filtered[, col] == 2)
                        mis <- sum(genotype.matrix.filtered[, col] == 3)
                        gt.anno[col, ] <- c(wt, het, hom, mis)
                    }

                    # Force Het, Hom, Missing in the right order
                    gt.anno$Total <- rowSums(gt.anno)
                    proportions <- gt.anno %>%
                        dplyr::mutate(across(c(WT, Het, Hom, Missing), ~ . / 
                                                Total * 100)) %>%
                        dplyr::select(-Total)
                    rownames(proportions) <- names 

                    colors.vaf <- circlize::colorRamp2(
                        c(0, 50, 100),
                        c("#414487FF", "#F6A97A", "#D44292")
                    )

                    # Create and render the Heatmap
                    Heatmap(
                        matrix = vaf.matrix.filtered.hm,
                        name = "VAF",
                        col = colors.vaf,
                        show_column_dend = TRUE,
                        show_row_dend = FALSE,
                        column_title = "Filtered Variants",
                        row_title = "Cells",
                        top_annotation = column_ha
                    )
                })
            })
        })

        ## "Continue with selection" analysis ----------------------------------
        # Reactive variable for current variants
        current_variants <- reactiveVal(NULL)
        kneeplot_data <- reactiveVal(NULL)
        print.var <- reactiveVal(NULL)
        print.clust <- reactiveVal(NULL)

        # Observer für den continue_var Button
        observeEvent(input$continue_var, {
            ## Elbow plot preparation ------------------------------------------
            req(selected_variants())
            # TODO maybe define reactive
            variant.ids.filtered.gene <- paste0(
                rowData(altExp(sce_filtered))$Gene,
                ":", rowData(altExp(sce_filtered))$id
            )

            # Update current_variants with the selected variants
            current_variants(selected_variants())
            current_variant_ids <- 
                sort(variant.ids.filtered.gene)[current_variants()]

            continue(TRUE)
            output$continue <- reactive({
                continue()
            })
            outputOptions(output, "continue", suspendWhenHidden = FALSE)

            kneeplot_data(plotElbow(sce_filtered, current_variant_ids))
        })


        # Render the kneeplot using the reactive variable
        output$kneeplot <- renderPlot({
            req(kneeplot_data()) # Ensure there is data available
            print(kneeplot_data()) # Render the plot
        })

        output$edgeplot <- renderPlot({
            # Compute the kneeplot data only upon clicking continue_var
            # Store the plot in the reactive variable
            variant.ids.filtered.gene <- paste0(
                rowData(altExp(sce_filtered))$Gene,
                ":", rowData(altExp(sce_filtered))$id
            )
            variants.of.interest <- 
                sort(variant.ids.filtered.gene)[current_variants()]

            # condition to run your specific branch, could be based on user 
            # input or button press
            plotDensityEdge(sce_filtered, variants.of.interest, input$minPts)
        })

        ## Clustering ----------------------------------------------------------
        # Define reactive variables for k2 and gg.clust
        k2 <- reactiveVal(NULL)
        gg.clust <- reactiveVal(NULL)
        ana_bar <- reactiveVal(NULL)
        vaf_hm <- reactiveVal(NULL)
        vaf_violin <- reactiveVal(NULL)
        vaf_map <- reactiveVal(NULL)
        method <- reactiveVal("kmeans")

        # Check input params
        # Consolidate all parameter checks into a single observer
        # Consolidate checks into a single observer
        observe({
            valid_n_clust <- is.numeric(input$n_clust) && input$n_clust >= 2
            valid_eps <- is.numeric(input$eps) && input$eps > 0 && 
                input$eps <= 1
            valid_resolution <- is.numeric(input$resolution) && 
                input$resolution > 0
            valid_minPts <- is.numeric(input$minPts) && input$minPts >= 2

            # Enable button based on all checks and provide informative feedback
            if (valid_n_clust && valid_eps && valid_resolution && 
                valid_minPts) {
                enable("kmeans_btn")
                output$error_message <- renderText("")
            } else {
                disable("kmeans_btn")
                output$error_message <- 
                    renderText("Please enter valid values for all parameters.")
            }
        })


        # Observe changes in the radio button and update the method
        observeEvent(input$radio, {
            method(input$radio)
        })


        # Observe the k-means button event
        observeEvent(input$kmeans_btn, {
            req(current_variants()) # Ensure variants are selected
            req(is.numeric(input$n_clust) && input$n_clust >= 2)
            req(method())
            method <- method()

            variant.ids.filtered.gene <- paste0(
                rowData(altExp(sce_filtered))$Gene,
                ":", rowData(altExp(sce_filtered))$id
            )
            variants.of.interest <- 
                sort(variant.ids.filtered.gene)[current_variants()]

            # Print selected variants
            print.var(paste0(
                "<ul>",
                paste0(
                    "<li>",
                    variants.of.interest,
                    "</li>",
                    collapse = ""
                ),
                "</ul>"
            ))

            # Print clustering
            try(
                {
                    print.clust(paste0(
                        "<ul>",
                        paste0(
                            "<li>",
                            "clustering: ",
                            as.character(method),
                            "</li>",
                            collapse = ""
                        ),
                        "</ul>"
                    ))
                },
                silent = TRUE
            )

            plots_visible_2(TRUE)

            ### Cluster plot ---------------------------------------------------
            req(plots_visible_2())

            if (method == "kmeans") {
                cluster.res <- clusterVariantSelection(sce_filtered,
                    variants.of.interest =
                        variants.of.interest,
                    method = "k-means",
                    n.clust = input$n_clust
                )
            } else if (method == "leiden") {
                tryCatch({
                    cluster.res <- 
                        clusterVariantSelection(sce_filtered,
                                                variants.of.interest =
                                                    variants.of.interest,
                                                method = "leiden",
                                                resolution = input$resolution
                    )
                }, error = function(e) {
                    # Error handling
                    showNotification(paste("Error: ", e$message), 
                                    type = "error")
                })
            } else {
                cluster.res <- clusterVariantSelection(sce_filtered,
                    variants.of.interest,
                    method = "dbscan",
                    eps.value = input$eps,
                    min.pts = input$minPts
                )
            }

            k2(cluster.res[[1]])
            gg.clust(cluster.res[["clusterplot"]])

            outputOptions(output, "plots_visible_2", suspendWhenHidden = FALSE)
        })

        is_valid_ggplot <- function(gg_clust_obj) {
            inherits(gg_clust_obj, "ggplot")
        }

        # Create a reactive expression to check validity of gg.clust
        observeEvent(gg.clust(), {
            gg_clust_result <- gg.clust()
            if (is_valid_ggplot(gg_clust_result)) {
                plots_visible_3(TRUE) # Set visibility to TRUE if valid
            } else {
                plots_visible_3(FALSE) # Set visibility to FALSE otherwise
                showNotification("gg.clust did not return a valid ggplot object.
                        Plots will not be generated.", type = "error")
            }
        })

        # Render the cluster plot using the reactive variable
        output$cluster_plot <- renderPlot({
            req(plots_visible_2())
            req(gg.clust())

            validate(
                need(is_valid_ggplot(gg.clust()), paste0(gg.clust()[[1]]))
            )
            print(gg.clust())
        })

        # End of Explore variants Panel plots
        # Plot calls Explore variants ------------------------------------------
        output$k2_output <- renderPrint({
            req(k2())
            return(k2()) # Render the k-means result stored in k2
        })

        # Expose the visibility state
        output$plots_visible_2 <- reactive({
            plots_visible_2()
        })

        output$plots_visible_3 <- reactive({
            plots_visible_3()
        })

        outputOptions(output, "plots_visible_3", suspendWhenHidden = FALSE)

        output$vaf_hm <- renderPlot({
            req(k2())
            req(plots_visible_2())
            req(plots_visible_3())
            req(current_variants())

            variant.ids.filtered.gene <- paste0(
                rowData(altExp(sce_filtered))$Gene,
                ":", rowData(altExp(sce_filtered))$id
            )
            variants.of.interest <-
                sort(variant.ids.filtered.gene)[current_variants()]

            # Make colorpalette
            chromosomes <- c(paste0("chr", seq(1, 21)), "chrX", "chrY")
            colors.vaf <- circlize::colorRamp2(
                c(0, 50, 100),
                c("#414487FF", "#F6A97A", "#D44292")
            )

            vaf.matrix.filtered <- as.data.frame(t(assay(altExp(
                sce_filtered,
                "variants"
            ), "VAF")))
            colnames(vaf.matrix.filtered) <-
                paste0(
                    rowData(altExp(sce_filtered, "variants"))$Gene,
                    ":", rowData(altExp(
                        sce_filtered,
                        "variants"
                    ))$id
                )

            genotype.matrix.filtered <- as.data.frame(t(assay(
                altExp(sce_filtered),
                "Genotype"
            )))
            colnames(genotype.matrix.filtered) <-
                paste0(
                    rowData(altExp(sce_filtered))$Gene, ":",
                    rowData(altExp(sce_filtered))$id
                )

            vaf.matrix.filtered.hm <- 
                vaf.matrix.filtered[, variants.of.interest]
            column_ha <- HeatmapAnnotation(
                chr = factor(
                    str_extract(
                        colnames(vaf.matrix.filtered.hm),
                        "chr(\\d|X|Y)+"
                    ),
                    levels = chromosomes
                ),
                col = list(chr = chr_palette)
            )
            collect <- data.frame(row.names = "")

            # GT matrix annotation
            df <- do.call(
                rbind,
                lapply(
                    genotype.matrix.filtered[, variants.of.interest],
                    function(x) {
                        length(x) <- 4
                        return(x)
                    }
                )
            )

            gt.anno <- data.frame(
                WT = integer(),
                Het = integer(),
                Hom = integer(),
                Missing = integer()
            )

            for (col in seq(1, ncol(genotype.matrix.filtered))) {
                wt <- sum(genotype.matrix.filtered[, col] == 0)
                het <- sum(genotype.matrix.filtered[, col] == 1)
                hom <- sum(genotype.matrix.filtered[, col] == 2)
                mis <- sum(genotype.matrix.filtered[, col] == 3)
                gt.anno[col, ] <- c(wt, het, hom, mis)
            }

            gt.anno$Total <- rowSums(gt.anno)
            proportions <- gt.anno %>%
                dplyr::mutate(across(c(WT, Het, Hom, Missing), ~ . / 
                                        Total * 100)) %>%
                dplyr::select(-Total)
            rownames(proportions) <- variant.ids.filtered.gene

            # Genotype annotation
            anno.bar <- anno_barplot(proportions,
                bar_width = 1, height = unit(3, "cm"),
                gp = gpar(fill = c(
                    WT = "#414487FF",
                    Het = "#F6A97A",
                    Hom = "#D44292",
                    Missing = "grey"
                ))
            )
            # Add cluster annoation
            colors <- gg_color_hue(length(unique(gg.clust()$data$cluster)))
            color_palette <- 
                setNames(colors, as.character(seq(1, length(colors))))
            row_annot <- rowAnnotation(
                cluster = as.factor(gg.clust()$data$cluster),
                col = list(cluster = color_palette)
            )
            vaf_hm(Heatmap(
                matrix = vaf.matrix.filtered.hm,
                name = "VAF",
                col = colors.vaf,
                show_column_dend = TRUE,
                show_row_dend = FALSE,
                column_title = "Filtered Variants",
                row_title = "Cells",
                top_annotation = column_ha,
                left_annotation = row_annot,
                row_split = as.factor(gg.clust()$data$cluster)
            ))
            print(vaf_hm())
        })

        output$vaf_violin <- renderPlot({
            ## Violin: Explore variants ----------------------------------------
            req(k2())
            req(plots_visible_2())
            req(plots_visible_3())

            req(current_variants())

            variant.ids.filtered.gene <- paste0(
                rowData(altExp(sce_filtered))$Gene,
                ":", rowData(altExp(sce_filtered))$id
            )
            variants.of.interest <-
                sort(variant.ids.filtered.gene)[current_variants()]

            violin <- plotClusterVAF(sce_filtered,
                variants.of.interest = variants.of.interest,
                gg.clust = gg.clust()
            ) +
                theme(
                    title = element_text(size = 20),
                    text = element_text(size = 16)
                )
            vaf_violin(violin)
            print(vaf_violin())
        })

        output$vaf_map <- renderPlot({
            ## Map: Explore variants colored by VAF ----------------------------
            req(plots_visible_2())
            req(plots_visible_3())
            req(current_variants())

            variant.ids.filtered.gene <- paste0(
                rowData(altExp(sce_filtered))$Gene,
                ":", rowData(altExp(sce_filtered))$id
            )
            variants.of.interest <-
                sort(variant.ids.filtered.gene)[current_variants()]

            vaf_map(plotClusterVAFMap(sce_filtered,
                variants.of.interest = variants.of.interest,
                gg.clust = gg.clust()
            ) +
                theme(
                    title = element_text(size = 20),
                    text = element_text(size = 16)
                ))
            print(vaf_map())
        })

        output$ana_bar <- renderPlot({
            ## Bar: Explore variants -------------------------------------------
            req(plots_visible_2())
            req(plots_visible_3())
            req(current_variants())

            variant.ids.filtered.gene <- paste0(
                rowData(altExp(sce_filtered))$Gene,
                ":", rowData(altExp(sce_filtered))$id
            )
            variants.of.interest <-
                sort(variant.ids.filtered.gene)[current_variants()]
            ana_bar(plotClusterGenotype(sce_filtered,
                variants.of.interest = variants.of.interest,
                gg.clust = gg.clust()
            ) +
                theme(
                    title = element_text(size = 20),
                    text = element_text(size = 16)
                ))

            print(ana_bar())
        })

        output$selected_rows_2 <- renderUI({
            HTML(print.var())
        })

        output$selected_method <- renderUI({
            HTML(print.clust())
        })
    })

    output$text1 <- renderText({
        paste("You have selected", input$var)
        paste("You have selected", input$radio)
    })


    # Plots --------------------------------------------------------------------
    # Sequencing plot 1
    output$seq_plot1 <- renderPlot({
        req(input$upload, sce())
        sce_obj <- sce()
        validate(need(input$upload, "Please, select a file to start"))

        metadata <- sce_obj@metadata

        plot(0, type = "n", axes = FALSE, ann = FALSE)
        sample_name <- metadata[["sample_name"]]
        n_cells <- metadata[["n_cells"]]

        mtext(n_cells, side = 3, line = -2, cex = 3, col = "#22A884FF")
        mtext("Number of cells", side = 3, line = -4, cex = 1.5)

        mtext(sample_name, side = 1, line = -4, cex = 3, col = "#22A884FF")
        mtext("Sample name", side = 1, line = -0.5, cex = 1.5)

        # Draw box
        box(which = "outer", lty = "solid", col = "grey")
    })

    # Sequencing plot 2
    output$seq_plot2 <- renderPlot({
        req(input$upload)
        sce_obj <- sce()
        validate(need(input$upload, "Please, select a file to start"))
        metadata <- sce_obj@metadata

        plot(0, type = "n", axes = FALSE, ann = FALSE)
        mtext(
            round(as.numeric(metadata[["n_read_pairs_mapped_to_cells"]]) /
                as.numeric(metadata[["n_cells"]]), digits = 2),
            side = 3, line = -2, cex = 3,
            col = "#22A484FF"
        )
        mtext("Mean read pairs mapped to cells per cell",
            side = 3,
            line = -4, cex = 1.5
        )

        mtext(
            round((as.numeric(metadata[["n_read_pairs_mapped_to_cells"]]) /
                as.numeric(metadata[["n_read_pairs"]]) * 100), digits = 2),
            side = 1, line = -4, cex = 3, col = "#22A484FF"
        )
        mtext("Read pairs assigned to cells (%)", side = 1, line = -2, 
            cex = 1.5)

        # Draw box
        box(which = "outer", lty = "solid", col = "grey")
    })

    output$panelUniformityText <- renderText({
        req(input$upload)
        sce_obj <- sce()
        validate(need(input$upload, "Please, select a file to start"))
        metadata <- sce_obj@metadata
        paste(
            "Panel Uniformity (%):",
            round(as.numeric(metadata[["avg_panel_uniformity"]]) * 100,
                digits = 2
            )
        ) # Combine label with dynamic value
    })

    # Sequencing log-log plot
    output$seq_plot3 <- renderPlot({
        logLogPlot(sce()) +
            theme(
                title = element_text(size = 20),
                text = element_text(size = 16)
            )
    })

    # Sequencing plot 4
    output$seq_plot4 <- renderPlotly({
        req(input$upload)
        sce_obj <- sce()
        metadata <- sce_obj@metadata %>%
            t() %>%
            as.data.frame() %>%
            select(
                n_bases_r1, n_bases_r1_q30, n_bases_r2, n_bases_r2_q30,
                n_cell_barcode_bases, n_cell_barcode_bases_q30
            ) %>%
            tidyr::pivot_longer(everything(),
                names_to = "Type",
                values_to = "value"
            ) %>%
            mutate(value = as.numeric(unlist(value)))

        metadata %>%
            mutate(
                Type = Type %>%
                    str_replace("n_bases_", "Bases ") %>%
                    str_replace("_q30", " Q30") %>%
                    str_replace("r(\\d)", "R\\1") %>%
                    str_replace(
                        "n_cell_barcode_bases",
                        "Cell barcode bases"
                    ),
                group = gsub(" Q30", "", Type),
            ) %>%
            ggplot(aes(x = group, y = value, fill = Type)) +
            geom_bar(stat = "identity", position = "stack") +
            theme_minimal() +
            scale_fill_manual(values = c(
                `Bases R1` = "#F66D7A",
                `Bases R1 Q30` = "#c00c1d",
                `Bases R2` = "dodgerblue",
                `Bases R2 Q30` = "#003898",
                `Cell barcode bases` = "#00BA38",
                `Cell barcode bases Q30` = "#00561a"
            )) +
            labs(x = "", y = "Number of bases")
    })

    # Panel plot 1
    output$panel_plot1 <- renderPlot({
        req(input$upload)
        sce_obj <- sce()
        metadata <- sce_obj@metadata

        plot(0, type = "n", axes = FALSE, ann = FALSE)
        mtext(metadata[["panel_name"]],
            side = 3, line = -2, cex = 3,
            col = "#22A884FF"
        )
        mtext("Panel used", side = 3, line = -4, cex = 1.5)

        # Print ean mapped reads per cell
        mtext(metadata[["n_amplicons"]],
            side = 1, line = -4, cex = 3,
            col = "#22A884FF"
        )
        mtext("Number of amplicons", side = 1, line = -2, cex = 1.5)
        box(which = "outer", lty = "solid", col = "grey")
    })

    # Panel plot 2
    output$panel_plot2 <- renderPlot({
        req(input$upload)
        sce_obj <- sce()
        metadata <- sce_obj@metadata

        genes <- vapply(
            str_split(rowData(sce_obj)$id, "_"), function(x) x[3],
            character(1)
        )
        plot(0, type = "n", axes = FALSE, ann = FALSE)
        mtext(length(unique(genes)), side = 3, line = -2, cex = 3, 
            col = "#F66D7A")
        mtext("Number of Genes covered", side = 3, line = -4, cex = 1.5)

        # Print ean mapped reads per cell
        mtext(
            floor(as.numeric(metadata[["n_read_pairs_mapped_to_cells"]]) /
                as.numeric(metadata[["n_cells"]]) /
                as.numeric(metadata[["n_amplicons"]])),
            side = 1, line = -4,
            cex = 3, col = "#22A484FF"
        )
        mtext("Average read pairs per \namplicon and cell",
            side = 1, line = -0.5,
            cex = 1.5
        )
        box(which = "outer", lty = "solid", col = "grey")
    })

    # Panel karyoplot
    output$panel_plot3 <- renderPlotly({
        plot <- plotAmpliconDistribution(sce = sce()) +
            theme(
                title = element_text(size = 20),
                text = element_text(size = 16)
            )
        plotly::ggplotly(plot@ggplot)
    })

    output$panel_plot4 <- renderPlot({
        plotNormalizedReadCounts(sce = sce()) +
            theme(
                title = element_text(size = 20),
                text = element_text(size = 16)
            )
    })

    output$panel_plot5 <- renderPlotly({
        plotPanelUniformity(sce = sce(), interactive = FALSE) +
            theme(
                title = element_text(size = 20),
                text = element_text(size = 16)
            )
    })

    # Tables -------------------------------------------------------------------
    output$data_table_sample <- renderDataTable({
        sce_obj <- sce()
        metadata <- sce_obj@metadata
        n.amp <- length(rownames(sce_obj)) %>%
            as.data.frame() %>%
            mutate(rowname = "n_amplions")
        metadata <- (metadata %>%
            unlist() %>%
            as.data.frame() %>%
            rownames_to_column())[c(24, 26, 7, 5, 25), ] %>%
            as.data.frame()
        metadata <- rbind(metadata, n.amp)
        metadata %>%
            mutate(rowname = gsub("n_", "Number of ", rowname)) %>%
            mutate(rowname = gsub("_", " ", rowname)) %>%
            mutate(rowname = str_to_title(rowname)) %>%
            datatable(.,
                extensions = "Buttons",
                options = list(
                    pageLength = 6,
                    dom = "Bt",
                    buttons = list(
                        list(
                            extend = "csv", filename =
                                paste0(
                                    "scafari_sequencing_sample_",
                                    metadata[["sample_name"]]
                                )
                        ),
                        list(
                            extend = "excel", filename =
                                paste0(
                                    "scafari_sequencing_sample_",
                                    metadata[["sample_name"]]
                                )
                        ),
                        list(
                            extend = "pdf", filename =
                                paste0(
                                    "scafari_sequencing_sample_",
                                    metadata[["sample_name"]]
                                )
                        ),
                        list(
                            extend = "copy", filename =
                                paste0(
                                    "scafari_sequencing_sample_",
                                    metadata[["sample_name"]]
                                )
                        )
                    )
                ),
                colnames = rep("", ncol(.)), rownames = FALSE
            )
    })

    output$data_table_sequencing <- renderDataTable({
        sce_obj <- sce()
        metadata <- sce_obj@metadata
        o <- "scafari_sequencing_overview_"
        rbind(
            "Total read pairs" = c(paste((
                as.numeric(metadata[["n_read_pairs"]])))),
            "Read pairs trimmed" =
                c(paste((as.numeric(metadata[["n_read_pairs_trimmed"]])))),
            "Read pairs with valid barcodes" =
                c(paste(round(as.numeric(
                    metadata[["n_read_pairs_valid_cell_barcodes"]]
                ))))
        ) %>%
            datatable(.,
                extensions = "Buttons",
                options = list(
                    pageLength = 5, width = "100%",
                    dom = "Bt",
                    buttons = list(
                        list(
                            extend = "csv",
                            filename = paste0(o, metadata[["sample_name"]])
                        ),
                        list(
                            extend = "excel", filename =
                                paste0(o, metadata[["sample_name"]])
                        ),
                        list(
                            extend = "pdf", filename =
                                paste0(o, metadata[["sample_name"]])
                        ),
                        list(
                            extend = "copy", filename =
                                paste0(o, metadata[["sample_name"]])
                        )
                    )
                ), colnames = NULL
            )
    })

    output$data_table_mapping <- renderDataTable({
        sce_obj <- sce()
        metadata <- sce_obj@metadata
        o <- "scafari_sequencing_mapping"
        rbind(
            "Reads mapped to genome (%)" =
                c(round(
                    (as.numeric(metadata[["n_reads_mapped"]]) /
                        (as.numeric(metadata[["n_read_pairs"]]) * 2) * 100),
                    digits = 2
                )),
            "Reads mapped to target (%)" =
                c(round(
                    (as.numeric(metadata[["n_reads_mapped_insert"]]) /
                        (as.numeric(metadata[["n_read_pairs"]]) * 2) * 100),
                    digits = 2
                )),
            "Average mapping error rate (%)" =
                c(round(as.numeric(metadata[["avg_mapping_error_rate"]]) * 100,
                    digits = 2
                ))
        ) %>%
            datatable(.,
                extensions = "Buttons",
                options = list(
                    pageLength = 5,
                    width = "100%",
                    dom = "Bt",
                    buttons = list(
                        list(
                            extend = "csv",
                            filename = paste0(
                                "scafari_sequencing_mapping",
                                metadata[["sample_name"]]
                            )
                        ),
                        list(
                            extend = "excel",
                            filename = paste0(
                                o,
                                metadata[["sample_name"]]
                            )
                        ),
                        list(
                            extend = "pdf",
                            filename = paste0(
                                o,
                                metadata[["sample_name"]]
                            )
                        ),
                        list(
                            extend = "copy",
                            filename = paste0(
                                o,
                                metadata[["sample_name"]]
                            )
                        )
                    )
                ),
                class = "display",
                colnames = rep("", ncol(.))
            )
    })

    # m.var$dp_cutoff
    # m.var$high_quality_variants
    # m.var$missing_cells_cutoff
    # m.var$missing_variants_cutoff
    # m.var$mutated_cells_cutoff
    output$data_table_tapestri <- renderDataTable({
        sce_obj <- sce()
        metadata <- sce_obj@metadata
        o <- "scafari_sequencing_tapestri_"
        rbind(
            "Depth cutoff" = c(paste(as.numeric(metadata[["dp_cutoff"]]))),
            "Missing cells cutoff" = c(paste(as.numeric(
                metadata[["missing_cells_cutoff"]]
            ))),
            "Missing variants cutoff" = c(paste(round(as.numeric(
                metadata[["missing_variants_cutoff"]]
            )))),
            "Mutated cells cutoff" = c(paste(as.numeric(
                metadata[["mutated_cells_cutoff"]]
            )))
        ) %>%
            datatable(.,
                extensions = "Buttons",
                options = list(
                    pageLength = 5, width = "100%",
                    dom = "Bt",
                    buttons = list(
                        list(
                            extend = "csv",
                            filename = paste0(o, metadata[["sample_name"]])
                        ),
                        list(
                            extend = "excel",
                            filename = paste0(o, metadata[["sample_name"]])
                        ),
                        list(
                            extend = "pdf",
                            filename = paste0(o, metadata[["sample_name"]])
                        ),
                        list(
                            extend = "copy",
                            filename = paste0(o, metadata[["sample_name"]])
                        )
                    )
                ), colnames = NULL
            )
    })

    ## Occurence of genes in panel ---------------------------------------------
    output$data_table_overview <- renderDataTable({
        temp_dir <- tempdir()
        known_canon_path <- file.path(
            temp_dir,
            "UCSC_hg19_knownCanonical_goldenPath.txt"
        )

        if (!file.exists(known_canon_path)) {
            url <- paste0(
                "http://hgdownload.soe.ucsc.edu/goldenPath/hg19/database/",
                "knownCanonical.txt.gz"
            )
            destfile <- file.path(
                temp_dir,
                "UCSC_hg19_knownCanonical_goldenPath.txt.gz"
            )

            withProgress(message = "Downloading known canonicals", value = 0, {
                # Define a custom function to mimic progress
                download_with_progress <- function(url, destfile) {
                    download.file(url, destfile)
                }

                # Mimic the progress (for demonstration, we just have 1 step)
                incProgress(0.5) # Simulate 50% progress

                # Call the download function
                download_with_progress(url, destfile)

                incProgress(0.5) # Simulate reaching 100%
            })

            # Decompress the file
            try(R.utils::gunzip(destfile, remove = FALSE))
        }
        annotateAmplicons(sce = sce(), known.canon = known_canon_path, 
                        shiny = TRUE)
    })
}
