% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/processPCA.R
\name{processPCA}
\alias{processPCA}
\title{Process PCA for SingleCellExperiment Objects}
\usage{
processPCA(
  sce_object,
  assay_name = "logcounts",
  n_hvgs = 2000,
  max_cells = NULL
)
}
\arguments{
\item{sce_object}{A \code{\linkS4class{SingleCellExperiment}} object to process.}

\item{assay_name}{Name of the assay to use for HVG selection and PCA computation.
Should contain log-normalized expression values. Default is "logcounts".}

\item{n_hvgs}{Number of highly variable genes to select for PCA computation. Default is 2000.}

\item{max_cells}{Maximum number of cells to retain if downsampling is needed for PCA computation.
If NULL, no downsampling is performed. Default is NULL.}
}
\value{
A \code{\linkS4class{SingleCellExperiment}} object with valid PCA in the reducedDims slot,
including rotation matrix and percentVar attributes. Will have original cell count if PCA was valid,
or at most max_cells if PCA was computed.
}
\description{
This function ensures that a \code{\linkS4class{SingleCellExperiment}} object has valid PCA computed using
highly variable genes when needed. It only performs downsampling when PCA computation
is required, preserving existing valid PCA computations without modification.
}
\details{
The function performs the following operations:
\itemize{
  \item Checks if PCA exists and is valid in the provided \code{\linkS4class{SingleCellExperiment}} object
  \item Validates PCA integrity including rotation matrix, percentVar, gene consistency, and dimensions
  \item If PCA is valid, returns the object unchanged (no downsampling)
  \item If PCA is missing or invalid and dataset is large, downsamples before computing PCA
  \item Computes PCA using highly variable genes when PCA is missing or invalid
  \item Utilizes scran for HVG selection and scater for PCA computation (soft dependencies)
}

The downsampling strategy uses random sampling without replacement and only occurs
when PCA computation is necessary. This preserves expensive pre-computed PCA results
while ensuring computational efficiency for new PCA computations.

PCA validation includes checking for:
\itemize{
  \item Presence of PCA in reducedDims
  \item Existence of rotation matrix and percentVar attributes
  \item Gene consistency between rotation matrix and current assay
  \item Dimension consistency between PCA coordinates and cell count
}
}
\note{
This function requires the scran and scater packages for HVG selection and PCA computation.
These packages should be installed via BiocManager::install(c("scran", "scater")).

Objects with existing valid PCA are returned unchanged to preserve expensive pre-computations.
Only datasets requiring PCA computation are subject to downsampling.
}
\examples{
library(SingleCellExperiment)

# Load data
data("reference_data")
data("query_data")

# Example 1: Dataset without PCA (will compute PCA)
query_no_pca <- query_data
reducedDims(query_no_pca) <- list()  # Remove existing PCA

processed_query <- processPCA(sce_object = query_no_pca, n_hvgs = 500)
"PCA" \%in\% reducedDimNames(processed_query)  # Should be TRUE
ncol(processed_query)  # Should be 503 (unchanged)

# Example 2: Dataset with existing valid PCA (will be preserved)
processed_existing <- processPCA(sce_object = query_data, n_hvgs = 500)
ncol(processed_existing)  # Should be 503 (unchanged, no downsampling)

# Example 3: Large dataset requiring downsampling for PCA computation
ref_no_pca <- reference_data
reducedDims(ref_no_pca) <- list()  # Remove existing PCA

processed_large <- processPCA(sce_object = ref_no_pca,
                              n_hvgs = 800,
                              max_cells = 1000)
ncol(processed_large)  # Should be 1000 (downsampled for PCA computation)

# Example 4: Large dataset with existing PCA (no downsampling)
processed_large_existing <- processPCA(sce_object = reference_data,
                                       n_hvgs = 800,
                                       max_cells = 1000)
ncol(processed_large_existing)  # Should be 1500 (preserved, no downsampling)

}
\author{
Anthony Christidis, \email{anthony-alexander_christidis@hms.harvard.edu}
}
