% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/streaming_callbacks.R
\name{streaming_callbacks}
\alias{streaming_callbacks}
\title{Suggested setup for FASTQ streaming}
\usage{
streaming_callbacks(
  input_file,
  output_table_file,
  chunk_size = 1e+06,
  verbose = TRUE,
  min_width = NULL
)
}
\arguments{
\item{input_file}{A character vector containing the paths to the FASTQ files
to be used for demultiplexing. Often this is only one file, but multiple
files are supported such that demultiplexing data from multiple lanes does
not require merging the lanes first.}

\item{output_table_file}{The path to which the output
barcode table will be written.}

\item{chunk_size}{Integer, the number of reads to process in each chunk.}

\item{verbose}{Logical scalar: Should the progress be displayed?}

\item{min_width}{Optional integer scalar:
Minimum width of the sequences to keep.
For reads which are shorter than this, a warning is emitted and the
reads are removed and ignored and thus not appear in any statistics.
The data loader is \strong{not} supposed to be used as a length filter,
so this option
is more like an escape hatch for being able to deal
with sequences which have not been
properly filtered beforehand.}
}
\value{
A list with the following elements, all of which are intended to be
used as the corresponding arguments to \code{\link[=streaming_demultiplex]{streaming_demultiplex()}}:
\itemize{
\item \code{state_init}
\item \code{loader}
\item \code{archiver}
}
}
\description{
Even though the user can define the arguments \code{state_init},
\code{loader}, and \code{archiver}
for \code{\link[=streaming_demultiplex]{streaming_demultiplex()}}, this approach is only recommended
for advanced users. This functions defines a premade combinations of
these three arguments which should be suitable in most cases.
The loader streams a FASTQ file (having multiple files is also supported)
and the archiver outputs
a data frame to file consisting of the read name (\code{read}),
the sequences of all payloads (e.g. \code{UMI}), and barcode assignments
(\code{c('bc3','bc2','bc1')}).
}
\details{
If the read names have any spaces in them,
the loader will only keep the portion of
the read name preceding the first space. This is due to the Illumina
platform's behavior of encoding the sequencing direction (forward or reverse)
past the space.
Keeping the read names with the space is usually not desirable as it makes
the resulting barcode table more confusing and makes it more difficult to
group the forward and reverse reads together afterwards.
}
\examples{
library(purrr)
library(Biostrings)
input_fastq <- system.file(
    "extdata", "PETRI-seq_forward_reads.fq.gz", package = "posDemux")
output_barcode_table <- tempfile(pattern = "barcode_table", fileext = ".txt")

callbacks <- streaming_callbacks(
    input_file = input_fastq, output_table_file = output_barcode_table,
    chunk_size = 10000, verbose = TRUE)
barcode_files <- system.file(
    "extdata/PETRI-seq_barcodes",
    c(bc1 = "bc1.fa", bc2 = "bc2.fa", bc3 = "bc3.fa"),
    package = "posDemux"
    )
names(barcode_files) <- paste0("bc", 1L:3L)
barcode_index <- map(barcode_files, readDNAStringSet)
barcodes <- barcode_index[c("bc3", "bc2", "bc1")]
sequence_annotation <- c(UMI = "P", "B", "A", "B", "A", "B", "A")
segment_lengths <- c(7L, 7L, 15L, 7L, 14L, 7L, NA_integer_)
streaming_summary_res <- streaming_demultiplex(
    state_init = callbacks$state_init, loader = callbacks$loader,
    archiver = callbacks$archiver, barcodes = barcodes, allowed_mismatches = 1L,
    segments = sequence_annotation, segment_lengths = segment_lengths
    )
}
\seealso{
\code{\link[ShortRead:Sampler-class]{ShortRead::FastqStreamer()}} which is used
as a backend for the data loader.
}
