% Generated by roxygen2 (4.0.0): do not edit by hand
\name{normalizeArray}
\alias{NormalizeArray}
\alias{normalizeArray}
\title{Normalize tiling array data using sequence information}
\usage{
normalizeArray(peptideSet, method = "ZpepQuad", robust = TRUE,
  centered = TRUE)
}
\arguments{
\item{peptideSet}{A \code{peptideSet}. The expression data for the peptides as
well as annotations and ranges.}

\item{method}{A \code{character}. The normalization method to be used. Can be
"Zpep" or "ZpepQuad".}

\item{robust}{A \code{logical}. If TRUE, reweigthed least-squares estimates are
computed.}

\item{centered}{A \code{logical}. If TRUE, recenter the data.}
}
\value{
A \code{peptideSet} object with updated normalized intensity values.
}
\description{
This function is used to normalize the peptide microarray data using sequence
information.
}
\details{
The available methods are "Zpep" and "ZpepQuad". These methods fit a linear model
using either linear or linear and quadratic terms (respectively), regressing
intensity on the peptides' five Z-scale scores. A peptide Z-scale score is
obtained by summing over the Z-scale values in Sandburg et al (1998) of the amino
acids the peptide comprises.

Peptide Z-scale scores may be provided in the featureRange slot of peptideSet.
This slot is a \code{GRanges} object x, and the function will seek five columns
labelled z1 through z5 in values(x). If these are not found, the function attempts
to calculate Z-scales from sequence information found in peptide(peptideSet)

If robust = TRUE the linear model is fit with t_4 distributed errors. The method
returns the residuals of each peptide intensity in the fitted linear model. If
centered = TRUE the fitted intercept term is added back to the residuals of the fit.
}
\examples{
## This example curated from the vignette -- please see vignette("pepStat")
## for more information
if (require("pepDat")) {

  ## Get example GPR files + associated mapping file
  dirToParse <- system.file("extdata/gpr_samples", package = "pepDat")
  mapFile <- system.file("extdata/mapping.csv", package = "pepDat")

  ## Make a peptide set
  pSet <- makePeptideSet(files = NULL, path = dirToParse,
                         mapping.file = mapFile, log=TRUE)

  ## Plot array images -- useful for quality control
  plotArrayImage(pSet, array.index = 1)
  plotArrayResiduals(pSet, array.index = 1, smooth = TRUE)

  ## Summarize peptides, using pep_hxb2 as the position database
  data(pep_hxb2)
  psSet <- summarizePeptides(pSet, summary = "mean", position = pep_hxb2)

  ## Normalize the peptide set
  pnSet <- normalizeArray(psSet)

  ## Smooth
  psmSet <- slidingMean(pnSet, width = 9)

  ## Make calls
  calls <- makeCalls(psmSet, freq = TRUE, group = "treatment",
                     cutoff = .1, method = "FDR", verbose = TRUE)

  ## Produce a summary of the results
  summary <- restab(psmSet, calls)

}
}
\author{
Raphael Gottardo, Gregory Imholte
}
\references{
Sandberg, M., Eriksson, L., Jonsson, J., Sjostrom, M., and Wold,
S. (1998). New chemical descriptors relevant for the design of biologically
active peptides. A multivariate characterization of 87 amino acids. Journal of
 Medicinal Chemistry 41, 2481-2491.
}
\seealso{
\code{\link{summarizePeptides}}, \code{\link{makeCalls}}
}

