##
### -----------------------------------------------------------------------###
### Prepare data for testing
### -----------------------------------------------------------------------###
##
set.seed(1234)
##
required_colnames <- c(
  "chrom", "start", "end", "name", "score", "strand",
  "center", "sample_name"
)
##
data(syn_data_tibble, package = "peakCombiner")
test_data <- syn_data_tibble

input_colnames <- colnames(test_data)
##
test_data_prepared <- peakCombiner::prepareInputRegions(
  data = test_data,
  outputFormat = "tibble",
  showMessages = FALSE
)
test_data_center_expand <- peakCombiner::centerExpandRegions(
  data = test_data_prepared,
  centerBy = "center_column",
  outputFormat = "tibble",
  expandBy = NULL
)
##
input_colnames <- colnames(test_data_center_expand)
##
test_data_filtered <- peakCombiner::filterRegions(
  data = test_data_center_expand,
  includeByChromosomeName = NULL,
  excludeByBlacklist = NULL,
  includeAboveScoreCutoff = NULL,
  includeTopNScoring = NULL,
  outputFormat = "tibble"
)
##
result_colnames <- colnames(test_data_filtered)
##
test_data_combined <- peakCombiner::combineRegions(
  data = test_data_filtered,
  foundInSamples = 2,
  outputFormat = "tibble",
  combinedCenter = "nearest"
)
##
test_data_combined_ce <- peakCombiner::centerExpandRegions(
  data = test_data_combined,
  centerBy = "center_column",
  expandBy = NULL,
  outputFormat = "tibble"
)
##
### -----------------------------------------------------------------------###
### Test input
### -----------------------------------------------------------------------###
##
test_that("Input data frame has the expected structure", {
  data <- test_data_filtered
  ##
  expect_equal(length(input_colnames), 8)
  expect_identical(names(data), required_colnames)
  expect_true(is.character(data$chrom))
  expect_true(is.numeric(data$start))
  expect_true(is.numeric(data$end))
  expect_true(is.character(data$name))
  expect_true(is.numeric(data$score))
  expect_true(is.character(data$strand))
  expect_true(is.numeric(data$center))
  expect_true(is.character(data$sample_name))
  expect_true(sum(stringr::str_detect(data$name, "|")) > 0)
})
##
### -----------------------------------------------------------------------###
### Test Output
### -----------------------------------------------------------------------###
##
test_that("Output data frame is correct", {
  data <- test_data_filtered
  ##
  expect_setequal(colnames(data), required_colnames)
  expect_equal(ncol(data), 8)
  ##
  expect_identical(class(data)[2], "tbl")
  ##
  expect_true(is.character(data$chrom))
  expect_true(is.numeric(data$start))
  expect_true(is.numeric(data$end))
  expect_true(is.character(data$name))
  expect_true(is.numeric(data$score))
  expect_true(is.character(data$strand))
  expect_true(is.numeric(data$center))
  expect_true(is.character(data$sample_name))
  ##
  expect_equal(round(mean(data$center), 0), 299)
  expect_identical(nrow(data), 51L)
  expect_identical(data$start[1], 251)
  ##
  test_counts_left <- test_data_filtered |>
    dplyr::group_by(sample_name) |>
    dplyr::summarise(counts = dplyr::n()) |>
    dplyr::filter(sample_name == "treatment_rep1") |>
    dplyr::pull(counts)
  expect_identical(test_counts_left, 8L)
})
##
### --------------------------------------------------------------------------###
##
test_that("Output data frame is correct for data_prepared", {
  ##
  data <- test_data_prepared
  ##
  expect_no_error(peakCombiner::filterRegions(
    data = data,
    excludeByBlacklist = NULL,
    includeByChromosomeName = NULL,
    includeAboveScoreCutoff = NULL,
    includeTopNScoring = NULL,
    outputFormat = "tibble"
  ))
  ##
  result <- peakCombiner::filterRegions(
    data = data,
    excludeByBlacklist = NULL,
    includeByChromosomeName = NULL,
    includeAboveScoreCutoff = NULL,
    includeTopNScoring = NULL,
    outputFormat = "tibble"
  )
  ##
  expect_identical(nrow(result), 51L)
  expect_identical(result$start[9], 300)
})
##
test_that("Output data frame is correct for data_center_expand", {
  ##
  data <- test_data_center_expand
  ##
  result <- peakCombiner::filterRegions(
    data = data,
    excludeByBlacklist = NULL,
    includeByChromosomeName = NULL,
    includeAboveScoreCutoff = NULL,
    includeTopNScoring = NULL,
    outputFormat = "tibble"
  )
  ##
  expect_identical(nrow(result), 51L)
  expect_identical(result$start[9], 1)
})
##
test_that("Output data frame is correct for data_filtered", {
  ##
  data <- test_data_filtered
  result <- peakCombiner::filterRegions(
    data = data,
    excludeByBlacklist = NULL,
    includeByChromosomeName = NULL,
    includeAboveScoreCutoff = NULL,
    includeTopNScoring = NULL,
    outputFormat = "tibble"
  )
  ##
  expect_identical(nrow(result), 51L)
  expect_identical(result$start[2], 1)
})
##
test_that("Output data frame is correct for data_combined", {
  ##
  data <- test_data_combined
  result <- peakCombiner::filterRegions(
    data = data,
    excludeByBlacklist = NULL,
    includeByChromosomeName = NULL,
    includeAboveScoreCutoff = NULL,
    includeTopNScoring = NULL,
    outputFormat = "tibble"
  )
  ##
  expect_identical(nrow(result), 5L)
  expect_identical(result$start[5], 1)
})
##
test_that("Output data frame is correct for data_combined_ce", {
  ##
  data <- test_data_combined_ce
  result <- peakCombiner::filterRegions(
    data = data,
    excludeByBlacklist = NULL,
    includeByChromosomeName = NULL,
    includeAboveScoreCutoff = NULL,
    includeTopNScoring = NULL,
    outputFormat = "tibble"
  )
  ##
  expect_identical(nrow(result), 5L)
  expect_identical(result$start[2], 1)
})
##
### -----------------------------------------------------------------------###
