##
### -----------------------------------------------------------------------###
### Prepare data for testing
### -----------------------------------------------------------------------###
##
set.seed(1234)
##
required_colnames <- c(
  "chrom", "start", "end", "name", "score", "strand",
  "center", "sample_name"
)
##
data(syn_data_tibble, package = "peakCombiner")
test_data <- syn_data_tibble
input_colnames <- colnames(test_data)
##
test_data_prepared <- peakCombiner::prepareInputRegions(
  data = test_data,
  outputFormat = "tibble",
  showMessages = FALSE
)
##
test_data_center_expand <- peakCombiner::centerExpandRegions(
  data = test_data_prepared,
  centerBy = "center_column",
  expandBy = NULL,
  outputFormat = "tibble"
)
##
input_colnames <- colnames(test_data_center_expand)
##
keep_chromosomes <- c("chr1", "chr10", "chr42")
##
test_data_filtered <- peakCombiner:::filterByChromosomeNames(
  data = test_data_center_expand,
  includeByChromosomeName = keep_chromosomes
)
##
result_colnames <- colnames(test_data_filtered)
##
### -----------------------------------------------------------------------###
### Test input
### -----------------------------------------------------------------------###
##
test_that("Test if function works with correct input", {
  expect_no_error(peakCombiner:::filterByChromosomeNames(
    data = test_data_center_expand,
    includeByChromosomeName = keep_chromosomes
  ))
})
##
### -----------------------------------------------------------------------###
##
test_that("Input data frame has the expected structure", {
  data <- test_data_filtered
  ##
  expect_equal(length(input_colnames), 8)
  expect_identical(names(data), required_colnames)
  expect_true(is.character(data$chrom))
  expect_true(is.numeric(data$start))
  expect_true(is.numeric(data$end))
  expect_true(is.character(data$name))
  expect_true(is.numeric(data$score))
  expect_true(is.character(data$strand))
  expect_true(is.numeric(data$center))
  expect_true(is.character(data$sample_name))
  expect_true(sum(stringr::str_detect(data$name, "|")) > 0)
})
##
### -----------------------------------------------------------------------###
##
test_that("Required parameter 'filterByChromosomeNames' has expected
          structure", {
  expect_no_error(peakCombiner:::filterByChromosomeNames(
    data = test_data_filtered,
    includeByChromosomeName = NULL
  ))
  expect_no_error(peakCombiner:::filterByChromosomeNames(
    data = test_data_filtered,
    includeByChromosomeName = "chr1"
  ))
  expect_no_error(peakCombiner:::filterByChromosomeNames(
    data = test_data_filtered,
    includeByChromosomeName = keep_chromosomes
  ))
  ##
  expect_error(peakCombiner:::filterByChromosomeNames(
    data = test_data_filtered,
    includeByChromosomeName = NA
  ))
})
##
### -----------------------------------------------------------------------###
### Test Output
### -----------------------------------------------------------------------###
##
test_that("Output data frame is correct", {
  data <- test_data_filtered
  ##
  expect_setequal(colnames(data), required_colnames)
  expect_equal(ncol(data), 8)
  ##
  expect_identical(class(data)[2], "tbl")
  ##
  expect_true(is.character(data$chrom))
  expect_true(is.numeric(data$start))
  expect_true(is.numeric(data$end))
  expect_true(is.character(data$name))
  expect_true(is.numeric(data$score))
  expect_true(is.character(data$strand))
  expect_true(is.numeric(data$center))
  expect_true(is.character(data$sample_name))
  ##
  expect_equal(round(mean(data$center), 0), 290)
  expect_identical(nrow(data), 37L)
  expect_identical(data$start[1], 1)
})
##
### -----------------------------------------------------------------------###
