% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/nucleoSimFunctions.R
\name{syntheticNucMapFromDist}
\alias{syntheticNucMapFromDist}
\title{Generate a synthetic nucleosome map containing complete sequences}
\usage{
syntheticNucMapFromDist(
  wp.num,
  wp.del,
  wp.var,
  fuz.num,
  fuz.var,
  max.cover = 100,
  nuc.len = 147,
  len.var = 10,
  lin.len = 20,
  rnd.seed = NULL,
  as.ratio = FALSE,
  distr = c("Uniform", "Normal", "Student")
)
}
\arguments{
\item{wp.num}{a non-negative \code{integer}, the number of well-positioned
(non-overlapping) nucleosomes.}

\item{wp.del}{a non-negative \code{integer}, the number of well-positioned
nucleosomes to remove to create uncovered regions.}

\item{wp.var}{a non-negative \code{integer}, the variance associated with
the distribution used to assign the start position to the sequences of the
well-positioned nucleosomes. This parameter introduces some variation in
the starting positions.}

\item{fuz.num}{a non-negative \code{integer}, the number of
fuzzy nucleosomes. Those nucleosomes are
distributed accordingly to an uniform distribution all over the region.
Those nucleosomes can overlap other well-positioned or fuzzy nucleosomes.}

\item{fuz.var}{a non-negative \code{integer}, the variance associated with
the distribution used to assign the start position to the sequences of the
fuzzy nucleosomes. This parameter introduces some variation in
the starting positions.}

\item{max.cover}{a positive \code{integer}, the maximum coverage for
one nucleosome. The final coverage
can have a higher value than \code{max.cover} since sequences from
different nucleosomes can be overlapping. Default = 100.}

\item{nuc.len}{a non-negative \code{numeric}, the nucleosome length.
Default = 147.}

\item{len.var}{a non-negative \code{integer}, the variance associated to
the normal distribution used to add some variance to the length of each
sequence. Default = 10.}

\item{lin.len}{a non-negative \code{integer}, the length of
the DNA linker. Default = 20.}

\item{rnd.seed}{a single value, interpreted as an \code{integer}, or
\code{NULL}. If a \code{integer} is given, the value is used to set the seed
of the random number generator. By fixing the seed, the generated results
can be reproduced. Default = \code{NULL}.}

\item{as.ratio}{a \code{logical}, if \code{TRUE}, a synthetic naked DNA
control map is created and the ratio between it and the nucleosome coverage
are calculated. It can be used to simulate hybridization ratio data, like
the one in Tiling Arrays. Both control map and obtained ratio are
returned. Default = \code{FALSE}.}

\item{distr}{the name of the distribution used to generate the nucleosome
map. The choices are : \code{"Uniform"}, \code{"Normal"} and
\code{"Student"}. Default = \code{"Uniform"}.}
}
\value{
an \code{list} of \code{class} "syntheticNucMap" containing the
following elements:
\itemize{
    \item \code{call} the matched call.
    \item \code{wp.starts} a \code{vector} of \code{integer}, the start
positions of all well-positioned nucleosome regions. The central
position of the nucleosome is calculated as wp.starts + round(nuc.len/2).
    \item \code{wp.nreads} a \code{vector} of \code{integer}, the number of
sequences associated to each well-positioned nucleosome.
    \item \code{wp.reads} a \code{IRanges} containing the well-positioned
nucleosome sequences.
    \item \code{fuz.starts} a \code{vector} of \code{integer}, the
start position of all the fuzzy nucleosomes.
    \item \code{fuz.nreads} a \code{vector} of \code{integer}, the number
of sequences associated to each fuzzy nucleosome.
    \item \code{fuz.reads} a \code{IRanges} containing the fuzzy nucleosome
sequences.
    \item \code{syn.reads} a \code{IRanges} containing all the synthetic
nucleosome sequences (from both fuzzy and well-positioned nucleosomes).
    \item \code{nuc.len} a \code{numeric} the nucleosome length.
}
The following elements will be only returned if \code{as.ratio=TRUE}:
\itemize{
    \item \code{ctr.reads} a \code{IRanges} containing the naked DNA
(control) sequences.
    \item \code{syn.ratio} a \code{Rle} containing the calculated ratio
between the nucleosome coverage and the control coverage.
}
}
\description{
Generate a synthetic nucleosome map, a map with complete
sequences covering the nucleosome regions, using the distribution
selected by the user. The distribution is used to assign the start
position to the sequences associated with the nucleosomes. The user has
choice between three different distributions: Normal, Student and Uniform.

The synthetic nucleosome map creation is separated into 3 steps :

1. Adding well-positioned nucleosomes following specified parameters. The
nucleosomes are all positioned at equidistance. Assigning sequences of
variable length to each nucleosome using a normal distribution
and specified variance.

2. Deleting some well-positioned nucleosomes following specified parameters.
Each nucleosome has an equal probability to be selected.

3. Adding fuzzy nucleosomes following an uniform distribution ad specified
parameters. Assigning sequences of variable length to each nucleosome
using the specified
distribution and parameters. The sequence length is always following a
normal distribution.

This function is a modified version of the syntheticNucMap() function
from Bioconductor nucleR package (Flores and Orozco, 2011).
}
\examples{

## Generate a synthetic map with 20 well-positioned nucleosomes and 10 fuzzy
## nucleosomes using a Normal distribution with a variance of 30 for the
## well-positioned nucleosomes, a variance of 40 for the fuzzy nucleosomes
## and a seed of 15.
syntheticNucMapFromDist(wp.num = 20, wp.del = 0, wp.var = 30,
    fuz.num = 10, fuz.var = 40, rnd.seed = 15,
    distr = "Normal")

## Same output but with ratio
syntheticNucMapFromDist(wp.num = 20, wp.del = 0, wp.var = 30,
    fuz.num = 10, fuz.var = 40,
    rnd.seed = 15, as.ratio = TRUE, distr = "Normal")


}
\author{
Rawane Samb, Astrid Deschenes
}
