% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/methods.R
\docType{methods}
\name{diffR}
\alias{diffR}
\alias{diffR-generic}
\alias{diffr}
\alias{differenceCall}
\alias{DifferenceCalling}
\alias{diffR,integer,integer,GenomicRanges-method}
\alias{diffR,character,character,GenomicRanges-method}
\alias{diffR,character,character,data.frame-method}
\alias{diffR,character,character,character-method}
\title{Difference Calling between conditional ChIP-seq data in normR with diffR}
\usage{
diffR(treatment, control, genome, ...)

\S4method{diffR}{integer,integer,GenomicRanges}(treatment, control, genome,
  procs = 1L, verbose = TRUE, eps = 1e-05, iterations = 10,
  minP = 0.05)

\S4method{diffR}{character,character,GenomicRanges}(treatment, control, genome,
  countConfig = countConfigSingleEnd(), procs = 1L, verbose = TRUE,
  eps = 1e-05, iterations = 10, minP = 0.05)

\S4method{diffR}{character,character,data.frame}(treatment, control, genome,
  countConfig = countConfigSingleEnd(), procs = 1L, verbose = TRUE,
  eps = 1e-05, iterations = 10, minP = 0.05)

\S4method{diffR}{character,character,character}(treatment, control,
  genome = "", countConfig = countConfigSingleEnd(), procs = 1L,
  verbose = TRUE, eps = 1e-05, iterations = 10, minP = 0.05)
}
\arguments{
\item{treatment}{An \code{integer} vector of treatment counts or a
\code{character} pointing to the treatment bam file. In the latter case an
"\code{treatment}.bai" index file should exist in the same folder.}

\item{control}{An \code{integer} vector of control counts or a
\link{character} pointing to the control bam file. In the latter case an
"\code{control}.bai" index file should exist in the same folder.}

\item{genome}{Either \code{NULL} (default), a \code{character} specifying a
USCS genome identifier, a \link{data.frame} consisting of two columns or a
\link{GenomicRanges} specifying the genomic regions (see Details).}

\item{...}{Optional arguments for the respective implementations of
\code{\link{diffR}}.}

\item{procs}{An \code{integer} giving the number of parallel threads to
use.}

\item{verbose}{A \code{logical} indicating whether verbose output is
desired.}

\item{eps}{A \code{numeric} specifying the T Filter threshold and the
threshold for EM convergence, \emph{i.e.} the minimal difference in
log-likelihood in two consecutive steps.}

\item{iterations}{An \code{integer} specifying how many times the EM is
initialized with random model parameters.}

\item{minP}{An \code{integer} controlling the threshold for the T
method when filtering low power regions, i.e. regions with low counts.}

\item{countConfig}{A \code{\link{NormRCountConfig}} object specifying bam
counting parameters for read count retrieval. See Details.}
}
\value{
A \code{\link{NormRFit}} container holding results of the fit
with type \code{diffR}.
}
\description{
Difference calling between \code{treatment} (ChIP-seq 1) and \code{control}
(ChIP-seq 2) in normR is done by fitting background and two conditional
enrichments simultaenously.  Therefore, a mixture of three binomials is fit
to the data with Expectation Maximization (EM). After convergence of the EM,
the fitted background component is used to calculate significance for
treatment and control count pair. Based on this statistic, user can extract
significantly enriched/depleted regions in a condition with a desired
significance level.  These regions can be further analyzed within R or
exported (see \code{\link{NormRFit-class}}).  Furthermore, diffR
calculates a standardized conditional-specific enrichment given the
fitted background component. See also Details
}
\details{
Supplied count vectors for treatment and control should be of same length
and of type \code{integer}.

For convenience, read count vectors can be obtained directly from bam files.
In this case, please specify a bam file for treatment and control each and a
\code{genome}. Bam files should be indexed using samtools (\emph{i.e.}
samtools index file file.bai). Furthermore, bam files should contain a valid
header with given chromosome names. If \code{genome == NULL}(default),
chromosome names will be read from treatment bamheader. Please be aware that
bamheader might contain irregular contigs and chrM which influence the fit.
Also be sure that treatment and control contain the same chromosomes.
Otherwise an error will be thrown. If \code{genome} is a \code{character},
\code{\link[GenomeInfoDb]{fetchExtendedChromInfoFromUCSC}} is used to
resolve this to a valid UCSC genome identifier (see
\url{https://genome.ucsc.edu/cgi-bin/hgGateway} for available genomes). In
this case, only assembled molecules will be considered (no circular). Please
check if your bam files obey this annotation. If \code{genome} is a
\code{data.frame}, it represents the chromosome specification. The first
column will be used as chromosome ID and the second column will be used as
the chromosome lengths. If \code{genome} is a \code{GenomicRanges}, it
should contain the equally sized genomic loci to count in, e.g. promoters.
The binsize in the supplied NormRCountConfig is ignore in this case.

\code{bamCountConfig} is an instance of class \code{\link{NormRCountConfig}}
specifying settings for read counting on bam files. You can specify the
binsize, minimum mapping quality, shifting of read ends etc.. Please refer
to \code{\link{NormRFit-class}} for details.
}
\examples{
require(GenomicRanges)

### enrichR(): Calling Enrichment over Input
#load some example bamfiles
input <- system.file("extdata", "K562_Input.bam", package="normr")
chipK4 <- system.file("extdata", "K562_H3K4me3.bam", package="normr")
#region to count in (example files contain information only in this region)
gr <- GRanges("chr1", IRanges(seq(22500001, 25000000, 1000), width = 1000))
#configure your counting strategy (see BamCountConfig-class)
countConfiguration <- countConfigSingleEnd(binsize = 1000,
                                           mapq = 30, shift = 100)
#invoke enrichR to call enrichment
enrich <- enrichR(treatment = chipK4, control = input,
                  genome = gr,  countConfig = countConfiguration,
                  iterations = 10, procs = 1, verbose = TRUE)
#inspect the fit
enrich
summary(enrich)

## Not run:
#write significant regions to bed
#exportR(enrich, filename = "enrich.bed", fdr = 0.01)
#write normalized enrichment to bigWig
#exportR(enrich, filename = "enrich.bw")
## End(**Not run**)

### diffR(): Calling differences between two conditions
chipK36 <- system.file("extdata", "K562_H3K36me3.bam", package="normr")
diff <- diffR(treatment = chipK36, control = chipK4,
              genome = gr,  countConfig = countConfiguration,
              iterations = 10, procs = 1, verbose = TRUE)
summary(diff)

### regimeR(): Identification of broad and peak enrichment
regime <- regimeR(treatment = chipK36, control = input, models = 3,
                  genome = gr,  countConfig = countConfiguration,
                  iterations = 10, procs = 1, verbose = TRUE)
summary(regime)
}
\seealso{
\code{\link{NormRFit-class}} for functions on accessing and
exporting the diffR fit. \code{\link{NormRCountConfig-class}} for
configuration of the read counting procedure (binsize, mapping quality,...).
}
\author{
Johannes Helmuth \email{helmuth@molgen.mpg.de}
}
