#######################################################################################
## Authors:
##   Florian Auer [florian.auer@informatik.uni-augsburg.de]
##
## History:
##   Created on 02 February 2017 by Auer
##     
## Description:
##   Contains functions to easily generate the ndex_api_config.r file from an yml file
##   This functions only should be used for package maintenance!
##    
## Dependencies:
##   yaml
#######################################################################################


#' Default header for the ndex_api_config.r file
#' @return character containing the header
#' @note only for package maintenance!
#' @keywords internal
#' @examples
#' NULL
ndex_conf_header = paste0(    "################################################################################\n",
                                "## Authors:\n",
                                "##   Florian Auer [florian.auer@informatik.uni-augsburg.de]\n",
                                "##\n",
                                "## History:\n",
                                "##   Created on 28 November 2016 by Auer\n",
                                "##   Switched to yml file on 02 Februar 2017 by Auer\n",
                                "##     \n",
                                "## Description:\n",
                                "##   Contains API configuration for connecting to NDEx servers\n",
                                "##     \n",
                                "## Note:\n",
                                "##   This file is automatically generated from a YAML file!\n",
                                "################################################################################\n",
                                "\n",
                                "\n",
                                "#' NDEx server api configuration\n",
                                "#' \n",
                                "#' This nested list contains the url and methods for accessing the NDEx server via its REST full api.\n",
                                "#' It contains specifications for NDEx server api version 1.3 and 2.0. The default api is specified by 'defaultVersion'.\n",
                                "#' If possible, the version 2.0 should be used.\n",
                                "#' Own configurations must contain a 'version' entry!\n",
                                "#' \n",
                                "#' @return Nested list resembling the NDEx server REST API structure\n", 
                                "#' @examples \n", 
                                "#' names(ndex_config$Version_2.0) \n", 
                                "#' @export\n", 
                                "ndex_config = ")


#' Translates a nested list (as provided by a yaml file) into R code defining the lists
#' @note only for package maintenance!
#'
#' @param obj list; Nested list to be translated. The allowed list elements are: list, character, numeric, logical
#' @param indent character (optional) (default: '    '[4 whitespaces]); Character(s) used for the indentation. Default: <tab>
#' @param indentShift character (optional) (default: ''); Shifts the block to the right by putting the characters before each line (mostly used in internal recursion)
#'
#' @return character; R code for generating the nested list
#'
#' @keywords internal
#' @examples
#' test = list(bla='some text',blubb=list(a='more text', version='2.0'),justANumber=123456)
#' #$bla
#' #[1] "some text"
#' #$blubb
#' #$blubb$a
#' #[1] "more text"
#' #$blubb$version
#' #[1] "2.0"
#' #$justANumber
#' #[1] 123456
#' #
#' #listToRCode(test)
#' #[1] "list(\n\tbla=\"some text\",\n\tblubb=list(\n\t\ta=\"more text\",\n\t\tversion=\"2.0\"\n\t)\n)"
listToRCode = function(obj, indent='    ', indentShift=''){
    newIndent = paste0(indentShift, indent)
    txts = character()
    for(n in names(obj)){
        if('list' %in% class(obj[[n]])){
            txts = c(txts, paste0(newIndent, n, '=', listToRCode(obj[[n]], indent = indent, indentShift = newIndent)))
        }else if('character' %in% class(obj[[n]])){
          if(length(obj[[n]])>1){
            txts = c(txts, paste0(newIndent, n, '= c("', paste0(obj[[n]], collapse = '", "'),'")'))
          }else{
            txts = c(txts, paste0(newIndent, n, '="', obj[[n]],'"'))
          }
        }else if('numeric' %in% class(obj[[n]])){
          txts = c(txts, paste0(newIndent, n, '=', obj[[n]]))
        }else if('logical' %in% class(obj[[n]])){
            txts = c(txts, paste0(newIndent, n, '=', obj[[n]]))
        }
    }
    txt =paste0('list(\n', paste(txts, collapse = ",\n"), '\n', indentShift, ')')
    return(txt)
}


##' Translates a YAML file to a R config script
##' Run this manually if you want to update the ndex_config
##' @note only for package maintenance!
##'
##' @param yamlFile character (default: 'R/ndex_api_config.yml'); input file in YAML format
##' @param rScriptFile character (default: 'R/ndex_api_config.r'); output file for the R script
##' @param defaultHeader character (optional) (default: ndex_conf_header); text that will be put in front of the R script
##'
##' @keywords internal
##' @examples
##' # yamlToRConfig('R/ndex_api_config.yml', 'R/ndex_api_config.r', ndex_conf_header)
##' NULL
#yamlToRConfig = function(yamlFile='R/ndex_api_config.yml', rScriptFile='R/ndex_api_config.r', defaultHeader=ndex_conf_header){
#  yamlObj = yaml::yaml.load_file(yamlFile)
#  rCodeTxt = paste0(defaultHeader, listToRCode(yamlObj))
#  outFile = file(rScriptFile)
#  writeLines(rCodeTxt, outFile)
#  close(outFile)
#}


####################################################
##   Unused stuff
####################################################

##' Convert an R object into a YAML string
##' same as yaml::as.yaml(obj)
##' @note only for package maintenance!
##'
##' @param obj list; Nested list to be translated. The allowed list elements are: list, character, numeric
##' @param indent character (optional); Character(s) used for the indentation. Default: <tab>
##' @param indentShift character (optional); Shifts the block to the right by putting the characters before each line (mostly used in internal recursion)
##'
##' @return character; R code for generating the nested list
##'
##' @keywords internal
##' @examples
##' \dontrun{test = list(bla='some text',blubb=list(a='more text', version='2.0'),justANumber=123456)
##' #$bla
##' #[1] "some text"
##' #$blubb
##' #$blubb$a
##' #[1] "more text"
##' #$blubb$version
##' #[1] "2.0"
##' #$justANumber
##' #[1] 123456
##' 
##' listToYaml(test)
##' #[1] "bla: some text\nblubb:\n  a: more text\n  version: '2.0'\njustANumber: 123456.0\n"
##' }
#listToYaml = function(obj, indent='  ', indentShift=''){
#  txts = character()
#  for(n in names(obj)){
#    if('list' %in% class(obj[[n]])){
#      txts = c(txts, paste0(indentShift, n, ':\n', listToYaml(obj[[n]], indent = indent, indentShift = paste0(indentShift, indent))))
#    }else if('character' %in% class(obj[[n]])){
#      if(length(obj[[n]])>1){
#        txts = c(txts, paste0(indentShift, n, ': '))
#        for(li in obj[[n]]){
#          txts = c(txts, paste0(indentShift, indent, n, '- "', obj[[n]],'"'))
#        }
#      }else{
#        txts = c(txts, paste0(indentShift, n, ': "', obj[[n]],'"'))
#      }
#    }else if('numeric' %in% class(obj[[n]])){
#      txts = c(txts, paste0(indentShift, n, ': ', obj[[n]]))
#    }
#  }
#  txt = paste(txts, collapse = "\n")
#  return(txt)
#}
