% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/detectCommunity.R
\name{detectCommunity}
\alias{detectCommunity}
\title{Detect the spatial community of each cell}
\usage{
detectCommunity(
  object,
  colPairName,
  size_threshold = 0,
  group_by = NULL,
  name = "spatial_community",
  cluster_fun = "louvain",
  BPPARAM = SerialParam()
)
}
\arguments{
\item{object}{a \code{SingleCellExperiment} or \code{SpatialExperiment}
object}

\item{colPairName}{single character indicating the \code{colPair(object)}
entry containing the neighbor information.}

\item{size_threshold}{single positive numeric that specifies the minimum 
number of cells per community. Defaults to 0.}

\item{group_by}{single character indicating that spatial community 
detection will be performed separately for all unique entries to 
\code{colData(object)[,group_by]}.}

\item{name}{single character specifying the name of the output saved in 
\code{colData(object)}. Defaults to "spatial_community".}

\item{cluster_fun}{single character specifying the function to use for 
community detection. Options are all strings that contain the suffix of an 
\code{igraph} community detection algorithm (e.g. "walktrap"). 
Defaults to "louvain".}

\item{BPPARAM}{a \code{\link[BiocParallel]{BiocParallelParam-class}} object
defining how to parallelize computations. Applicable when \code{group_by} is 
specified and defaults to \code{SerialParam()}. 
For reproducibility between runs, we recommend defining \code{RNGseed} in the
 \code{\link[BiocParallel]{BiocParallelParam-class}} object.}
}
\value{
returns an object of \code{class(object)} containing a new column 
entry to \code{colData(object)[[name]]}.
}
\description{
Function to detect the spatial community of each cell as 
proposed by \href{https://www.nature.com/articles/s41586-019-1876-x}{
Jackson et al., The single-cell pathology landscape of breast cancer, Nature, 
2020}. Each cell is clustered based on its interactions as defined by a 
spatial object graph.
}
\section{Spatial community detection procedure}{

1. Create an igraph object from the edge list stored in 
\code{colPair(object, colPairName)}.  

2. Perform community detection using the specified \code{cluster_fun} algorithm.  
  
3. Store the community IDs in a vector and replace all communities with 
a size smaller than \code{size_threshold} by NA.   

Optional steps: Specify \code{group_by} to perform spatial community 
detection separately for all unique entries to \code{colData(object)[,group_by]} 
e.g. for tumor and non-tumor cells.
}

\examples{
library(cytomapper)
library(BiocParallel)
data(pancreasSCE)

sce <- buildSpatialGraph(pancreasSCE, img_id = "ImageNb",
                        type = "expansion",
                        name = "neighborhood",
                        threshold = 20)

## Detect spatial community 
set.seed(22)
sce <- detectCommunity(sce, 
                      colPairName = "neighborhood", 
                      size_threshold = 10)

plotSpatial(sce,
            img_id = "ImageNb",
            node_color_by = "spatial_community",
            scales = "free")
            
## Detect spatial community - specify group_by
sce <- detectCommunity(sce, 
                       colPairName = "neighborhood", 
                       group_by = "CellType", 
                       size_threshold = 10,
                       BPPARAM = SerialParam(RNGseed = 22)) 

plotSpatial(sce,
            img_id = "ImageNb",
            node_color_by = "spatial_community",
            scales = "free")
           
}
\references{
\href{https://www.nature.com/articles/s41586-019-1876-x}{
Jackson et al., The single-cell pathology landscape of breast cancer, 
Nature, 2020}
}
\author{
Lasse Meyer (\email{lasse.meyer@uzh.ch})
}
