### Unit tests for methodsEmapInternal.R functions

library(enrichViewNet)
library(ggplot2)

data(parentalNapaVsDMSOEnrichment)
data(rosaNapaVsDMSOEnrichment)


### Tests validateCreateEnrichMapArguments() results

context("validateCreateEnrichMapArguments() results")

test_that("validateCreateEnrichMapArguments() must return expected result", {
    
    result <- enrichViewNet:::validateCreateEnrichMapArguments(
        gostObject=parentalNapaVsDMSOEnrichment, query="parental_napa_vs_DMSO", 
        source="GO:CC", termIDs=NULL, removeRoot=TRUE, 
        showCategory=30, categoryLabel=1, categoryNode=1, 
        line=1)
    
    expect_true(result)
})


### Tests validateCreateEnrichMapMultiBasicArgs() results

context("validateCreateEnrichMapMultiBasicArgs() results")

test_that("validateCreateEnrichMapMultiBasicArgs() must return expected result", {
    
    result <- enrichViewNet:::validateCreateEnrichMapMultiBasicArgs(
        gostObjectList=list(parentalNapaVsDMSOEnrichment, 
                                        rosaNapaVsDMSOEnrichment), 
        queryList=list("parental_napa_vs_DMSO", "rosa_napa_vs_DMSO"), 
        source="GO:CC", termIDs=NULL, removeRoot=TRUE, showCategory=30, 
        categoryLabel=1, categoryNode=1, line=1)
    
    expect_true(result)
})


### Tests validateCreateEnrichMapMultiBasicAsIgraphArgs() results

context("validateCreateEnrichMapMultiBasicAsIgraphArgs() results")

test_that("validateCreateEnrichMapMultiBasicAsIgraphArgs() must return expected result", {
    
    result <- enrichViewNet:::validateCreateEnrichMapMultiBasicAsIgraphArgs(
        gostObjectList=list(parentalNapaVsDMSOEnrichment, 
                            rosaNapaVsDMSOEnrichment), 
        queryList=list("parental_napa_vs_DMSO", "rosa_napa_vs_DMSO"), 
        source="GO:CC", termIDs=NULL, removeRoot=TRUE, showCategory=30, 
        similarityCutOff=0.2)
    
    expect_true(result)
})


### Tests validateCreateEnrichMapMultiBasicGOSTArgs() results

context("validateCreateEnrichMapMultiBasicGOSTArgs() results")

test_that("validateCreateEnrichMapMultiBasicGOSTArgs() must return expected result", {
    
    result <- enrichViewNet:::validateCreateEnrichMapMultiBasicGOSTArgs(
        gostObjectList=list(parentalNapaVsDMSOEnrichment, 
                            rosaNapaVsDMSOEnrichment), 
        queryList=list("parental_napa_vs_DMSO", "rosa_napa_vs_DMSO"), 
        source="GO:CC", termIDs=NULL, removeRoot=TRUE)
    
    expect_true(result)
})


### Tests validateCreateEnrichMapMultiComplexArg() results

context("validateCreateEnrichMapMultiComplexArg() results")

test_that("validateCreateEnrichMapMultiComplexArg() must return expected result", {
    
    queryDF <- data.frame(queryName=c("parental_napa_vs_DMSO", 
        "rosa_napa_vs_DMSO", "rosa_napa_vs_DMSO"), 
        source=c("GO:CC", "REAC", "GO:CC"), removeRoot=c(TRUE, TRUE, TRUE),
        groupName=c("parental GO:CC", "rosa Reactome", "rosa GO:CC"), 
        termIDs=c("", "", ""), stringsAsFactors=FALSE)
    
    result <- enrichViewNet:::validateCreateEnrichMapMultiComplexArg(
        gostObjectList=list(parentalNapaVsDMSOEnrichment, 
                rosaNapaVsDMSOEnrichment, rosaNapaVsDMSOEnrichment), 
        queryInfo=queryDF, showCategory=30, categoryLabel=1, 
        categoryNode=1, line=1)
    
    expect_true(result)
})


### Tests validateCreateEnrichMapSubSectionArguments() results

context("validateCreateEnrichMapSubSectionArguments() results")

test_that("validateCreateEnrichMapSubSectionArguments() must return expected result", {
    
    result <- enrichViewNet:::validateCreateEnrichMapSubSectionArguments(
        showCategory=30, categoryLabel=1, categoryNode=1,
        line=2)
    
    expect_true(result)
})


### Tests manageNameDuplicationInEmap() results

context("manageNameDuplicationInEmap() results")

test_that("manageNameDuplicationInEmap() must return expected result", {
    
    clustData <- data.frame(Cluster=c("group 1" , "group 1", "group 2", 
                                        "group 2", "group 2", "group 1"), 
        ID=c("WP:WP4925", "WP:WP382", "KEGG:04010",  "KEGG:01010", 
                "KEGG:919191", "KEGG:101010"),
        Description=c("Unfolded protein response", 
            rep("MAPK signaling pathway", 2), "VEGFA-VEGFR2 signaling", 
            rep("Human T-cell pathway", 2)),
        GeneRatio=c("4/157", "3/157", "3/157", rep("2/157", 3)),
        BgRatio=c("4/24022", "3/24022", "3/24022", rep("2/24022", 3)),
        pvalues=c(1.55e-4, 8.13e-8, 4.33e-5, 3.2e-5, 3.1e-5, 3.5e-5),
        p.adjust=c(1e-3, 1e-3, 1.4e-3, 1e-3, 1e-3, 1e-3), 
        qvalue=c(1e-3, 1e-3, 1.4e-3, 1e-3, 1e-3, 1e-3), 
        geneID=c("ENSG000107968/ENSG000120129/ENSG000123358/ENSG000158050",
            "ENSG000107968/ENSG000120129/ENSG000158050",
            "ENSG000107968/ENSG000120129/ENSG000158050", 
            "ENSG000120129/ENSG000158050", "ENSG000120129/ENSG000158050",
            "ENSG000120129/ENSG000158050"),
        Count=c(4, 3, 3, 2, 2, 2))
    
    expected <- c("Unfolded protein response", 
        "MAPK signaling pathway (WP:WP382)", 
        "MAPK signaling pathway (KEGG:04010)", 
        "VEGFA-VEGFR2 signaling",  "Human T-cell pathway (KEGG:919191)", 
        "Human T-cell pathway (KEGG:101010)")
    
    result <- enrichViewNet:::manageNameDuplicationInEmap(clProfDF=clustData)

    expect_equal(result$Description, expected)
})


### Tests manageQueryDuplicationInEmap() results

context("manageQueryDuplicationInEmap() results")

test_that("manageQueryDuplicationInEmap() must return expected result", {
    
    queryList <- list("parental_vs_DMSO", "rosa_vs_DMSO", "parental_vs_DMSO", 
                    "rosa_vs_DMSO", "parental_vs_Control", "rosa_vs_DMSO", 
                    "Mia_vs_DMSO", "parental_vs_Control", "rosa_vs_DMSO")
    
    expected <- list("parental_vs_DMSO (1)", "rosa_vs_DMSO (1)", 
        "parental_vs_DMSO (2)", "rosa_vs_DMSO (2)", "parental_vs_Control (1)", 
        "rosa_vs_DMSO (3)", "Mia_vs_DMSO", "parental_vs_Control (2)", 
        "rosa_vs_DMSO (4)")
    
    result <- enrichViewNet:::manageQueryDuplicationInEmap(queryList=queryList)
    
    expect_equal(result, expected)
})


### Tests createBasicEmap() results

context("createBasicEmap() results")

test_that("createBasicEmap() must return expected result", {
    
    gostResults <- as.data.frame(parentalNapaVsDMSOEnrichment$result)
    gostResults <- gostResults[which(gostResults$source == "KEGG"),]
    gostResults <- gostResults[which(gostResults$term_id != "KEGG:00000"),]
    
    backgroundGenes <- parentalNapaVsDMSOEnrichment$meta$query_metadata$queries[["parental_napa_vs_DMSO"]]
    
    set.seed(111)
    
    graphRes <- enrichViewNet:::createBasicEmap(gostResults=gostResults, 
        backgroundGenes=backgroundGenes, showCategory=30L, 
        categoryLabel=1, categoryNode=1, significantMethod="FDR", line=1)
    
    expect_true(is_ggplot(graphRes))
    
    expect_true(all(graphRes$data$name %in% gostResults$term_name))
    
    expect_true(all(graphRes$data$size[order(graphRes$data$size, 
        decreasing=TRUE)] == 
        gostResults$intersection_size[order(gostResults$intersection_size, 
                                                decreasing=TRUE)]))
})


### Tests createMultiEmap() results

context("createMultiEmap() results")

test_that("createMultiEmap() must return expected result when 2 different enrichment analyses", {
    
    gostResults1 <- as.data.frame(parentalNapaVsDMSOEnrichment$result)
    gostResults1 <- gostResults1[which(gostResults1$source == "KEGG"),]
    gostResults1 <- gostResults1[which(gostResults1$term_id != "KEGG:00000"),]
    
    gostResults2 <- as.data.frame(rosaNapaVsDMSOEnrichment$result)
    gostResults2 <- gostResults2[which(gostResults2$source == "KEGG"),]
    gostResults2 <- gostResults2[which(gostResults2$term_id != "KEGG:00000"),]
    
    queryList <- list("parental_napa_vs_DMSO", "rosa_napa_vs_DMSO")
    
    set.seed(111)
    
    graphRes <- enrichViewNet:::createMultiEmap(gostResults=list(gostResults1, 
        gostResults2), queryList=queryList, showCategory=30L, 
        categoryLabel=1, categoryNode=1, line=1)
    
    expect_true(is_ggplot(graphRes))
    
    expected_terms <- c("MAPK signaling pathway", 
        "Transcriptional misregulation in cancer", 
        "Parathyroid hormone synthesis, secretion and action",
        "Apoptosis", "Colorectal cancer", "TNF signaling pathway", 
        "p53 signaling pathway", "Human T-cell leukemia virus 1 infection", 
        "Non-alcoholic fatty liver disease", "Amphetamine addiction",
        "Estrogen signaling pathway", "Viral carcinogenesis",
        "IL-17 signaling pathway", "Fluid shear stress and atherosclerosis", 
        "Hepatitis B")
    
    expected_term_size <- c(14, 9, 6, 7, 6, 6, 5, 7, 7, 4, 5, 4, 3, 3, 3)
    expected_term_size <- expected_term_size[order(expected_term_size, 
                                                        decreasing=TRUE)]
    expect_true(all(graphRes$data$name %in% expected_terms))
})

test_that("createMultiEmap() must return expected result when same different enrichment analysis", {
    
    gostResults1 <- as.data.frame(parentalNapaVsDMSOEnrichment$result)
    gostResults1 <- gostResults1[which(gostResults1$source == "KEGG"),]
    gostResults1 <- gostResults1[which(gostResults1$term_id != "KEGG:00000"),]
    
    gostResults2 <- as.data.frame(parentalNapaVsDMSOEnrichment$result)
    gostResults2 <- gostResults2[which(gostResults2$source == "REAC"),]
    gostResults2 <- gostResults2[1:10,]
    
    queryList <- list("parental_napa_vs_DMSO (1)", "parental_napa_vs_DMSO (2)")
    
    set.seed(111)
    
    graphRes <- enrichViewNet:::createMultiEmap(gostResults=list(gostResults1, 
        gostResults2), queryList=queryList, showCategory=30L, 
        categoryLabel=1, categoryNode=1, line=1)
    
    expect_true(is_ggplot(graphRes))
    
    expected_terms <- c("MAPK signaling pathway", 
        "Transcriptional misregulation in cancer", 
        "Parathyroid hormone synthesis, secretion and action",
        "Apoptosis", "Colorectal cancer", "TNF signaling pathway", 
        "p53 signaling pathway", "Human T-cell leukemia virus 1 infection", 
        "Non-alcoholic fatty liver disease", "Amphetamine addiction",
        "Estrogen signaling pathway", "NGF-stimulated transcription",
        "Nuclear Events (kinase and transcription factor activation)",
        "Response of EIF2AK1 (HRI) to heme deficiency",                    
        "Signaling by NTRK1 (TRKA)", "FOXO-mediated transcription",                                     
        "Signaling by NTRKs", "Signal Transduction",                                              
        "ATF4 activates genes in response to endoplasmic reticulum  stress", 
        "Generic Transcription Pathway", "PERK regulates gene expression" )
    
    expected_term_size <- c(14, 8, 6, 7, 6, 6, 5, 7, 6, 4, 5, 9, 9, 6, 9, 
                                7, 9, 34, 5, 23, 5)
    expected_term_size <- expected_term_size[order(expected_term_size, 
                                                   decreasing=TRUE)]
    
    expect_true(all(graphRes$data$name %in% expected_terms))
})


### Tests validateCreateEnrichMapMultiComplexAsIgraphArg() results

context("validateCreateEnrichMapMultiComplexAsIgraphArg() results")

test_that("validateCreateEnrichMapMultiComplexAsIgraphArg() must return expected result", {
    
    gostObjectList <- list(parentalNapaVsDMSOEnrichment, 
                            parentalNapaVsDMSOEnrichment)
    queryList <- list("parental_napa_vs_DMSO (1)", "parental_napa_vs_DMSO (2)")
    
    queryData <- data.frame(queryName=c("parental_napa_vs_DMSO", 
        "parental_napa_vs_DMSO"), source=c("KEGG", "REAC"), 
        removeRoot=c(TRUE, TRUE), termIDs=c("", ""), 
        groupName=c("parental - KEGG", "parental - Reactome"), 
        stringsAsFactors=FALSE)
    
    expect_true(enrichViewNet:::validateCreateEnrichMapMultiComplexAsIgraphArg(
        gostObjectList=gostObjectList, queryInfo=queryData, showCategory=20, 
        similarityCutOff=0.2))
})


### Tests similarityJaccard() results

context("similarityJaccard() results")

test_that("similarityJaccard() must return expected result", {
    
    resultData <- data.frame(Description=c("Term 1", "Term 2", "Term 3",
                                            "Term 4", "Term 5"), 
            geneID=c(
                paste0("ENSG001/ENSG002/ENSG003/ENSG004/ENSG005"), 
                paste0("ENSG001/ENSG002/ENSG003"),
                paste0("ENSG011/ENSG012/ENSG013"),
                paste0("ENSG001/ENSG002/ENSG003"),
                paste0("ENSG003/ENSG006/ENSG007/ENSG008/ENSG011")))
    
    exp <- matrix(data=NA, ncol=5, nrow=5)
    exp[2,1] <- 3.0/5.0
    exp[3,1] <- 0.0
    exp[3,2] <- 0.0
    exp[4,1] <- 3.0/5.0
    exp[4,2] <- 1.0
    exp[4,3] <- 0.0
    exp[5,1] <- 1.0/9.0
    exp[5,2] <- 1.0/7.0
    exp[5,3] <- 1.0/7.0
    exp[5,4] <- 1.0/7.0
    
    colnames(exp) <- c("Term 1", "Term 2", "Term 3", "Term 4", "Term 5")
    rownames(exp) <- c("Term 1", "Term 2", "Term 3", "Term 4", "Term 5")
     
    result <- enrichViewNet:::similarityJaccard(resultDF=resultData)
    
    expect_equivalent(result, exp)
})

test_that("similarityJaccard() must return expected result when zero", {
    
    resultData <- data.frame(Description=c("Term 1", "Term 21", "Term 3",
                                            "Term 4", "Term 5"), 
            geneID=c(
                paste0("ENSG001/ENSG002/ENSG003/ENSG004/ENSG005"), 
                paste0("ENSG001/ENSG002/ENSG003"),
                paste0("ENSG011/ENSG012/ENSG013/ENSG001"),
                paste0(""),
                paste0("")))
    
    exp <- matrix(data=NA, ncol=5, nrow=5)
    exp[2,1] <- 3.0/5.0
    exp[3,1] <- 1.0/8.0
    exp[3,2] <- 1.0/6.0
    exp[4,1] <- 0.0
    exp[4,2] <- 0.0
    exp[4,3] <- 0.0
    exp[5,1] <- 0.0
    exp[5,2] <- 0.0
    exp[5,3] <- 0.0
    exp[5,4] <- 0.0
    
    colnames(exp) <- c("Term 1", "Term 21", "Term 3", "Term 4", "Term 5")
    rownames(exp) <- c("Term 1", "Term 21", "Term 3", "Term 4", "Term 5")
    
    result <- enrichViewNet:::similarityJaccard(resultDF=resultData)
    
    expect_equivalent(result, exp)
})

    