% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dittoPlotVarsAcrossGroups.R
\name{dittoPlotVarsAcrossGroups}
\alias{dittoPlotVarsAcrossGroups}
\title{Generates a dittoPlot where data points are genes/metadata summaries, per groups, instead of individual values per cells/samples.}
\usage{
dittoPlotVarsAcrossGroups(
  object,
  vars,
  group.by,
  color.by = group.by,
  split.by = NULL,
  summary.fxn = mean,
  cells.use = NULL,
  plots = c("vlnplot", "jitter"),
  assay = .default_assay(object),
  slot = .default_slot(object),
  adjustment = "z-score",
  swap.rownames = NULL,
  do.hover = FALSE,
  main = NULL,
  sub = NULL,
  ylab = "make",
  y.breaks = NULL,
  min = NA,
  max = NA,
  xlab = group.by,
  x.labels = NULL,
  x.labels.rotate = NA,
  x.reorder = NULL,
  groupings.drop.unused = TRUE,
  color.panel = dittoColors(),
  colors = c(seq_along(color.panel)),
  theme = theme_classic(),
  jitter.size = 1,
  jitter.width = 0.2,
  jitter.color = "black",
  jitter.position.dodge = boxplot.position.dodge,
  do.raster = FALSE,
  raster.dpi = 300,
  boxplot.width = 0.2,
  boxplot.color = "black",
  boxplot.show.outliers = NA,
  boxplot.outlier.size = 1.5,
  boxplot.fill = TRUE,
  boxplot.position.dodge = vlnplot.width,
  boxplot.lineweight = 1,
  vlnplot.lineweight = 1,
  vlnplot.width = 1,
  vlnplot.scaling = "area",
  vlnplot.quantiles = NULL,
  ridgeplot.lineweight = 1,
  ridgeplot.scale = 1.25,
  ridgeplot.ymax.expansion = NA,
  ridgeplot.shape = c("smooth", "hist"),
  ridgeplot.bins = 30,
  ridgeplot.binwidth = NULL,
  add.line = NULL,
  line.linetype = "dashed",
  line.color = "black",
  split.nrow = NULL,
  split.ncol = NULL,
  split.adjust = list(),
  legend.show = TRUE,
  legend.title = NULL,
  data.out = FALSE
)
}
\arguments{
\item{object}{A Seurat, SingleCellExperiment, or SummarizedExperiment object.}

\item{vars}{String vector (example: \code{c("gene1","gene2","gene3")}) which selects which variables, typically genes, to extract from the object, summarize across groups, and add to the plot}

\item{group.by}{String representing the name of a metadata to use for separating the cells/samples into discrete groups.}

\item{color.by}{String representing the name of a metadata to use for setting fills.
Great for highlighting subgroups when wanted, but it defaults to \code{group.by} so this input can be skipped otherwise.
Affects boxplot, vlnplot, and ridgeplot fills.}

\item{split.by}{1 or 2 strings naming discrete metadata to use for splitting the cells/samples into multiple plots with ggplot faceting.

When 2 metadatas are named, c(row,col), the first is used as rows and the second is used for columns of the resulting grid.

When 1 metadata is named, shape control can be achieved with \code{split.nrow} and \code{split.ncol}}

\item{summary.fxn}{A function which sets how variables' data will be summarized across the groups.
Default is \code{\link{mean}}, which will take the average value, but any function can be used as long as it takes in a numeric vector and returns a single numeric value.
Alternative examples: \code{\link{median}}, \code{\link{max}}, or \code{function(x) mean(x!=0)}.}

\item{cells.use}{String vector of cells'/samples' names OR an integer vector specifying the indices of cells/samples which should be included.

Alternatively, a Logical vector, the same length as the number of cells in the object, which sets which cells to include.}

\item{plots}{String vector which sets the types of plots to include: possibilities = "jitter", "boxplot", "vlnplot", "ridgeplot".
Order matters: c("vlnplot", "boxplot", "jitter") will put a violin plot in the back, boxplot in the middle, and then individual dots in the front.
See details section for more info.}

\item{assay, slot}{single strings or integers (SCEs and SEs) or an optionally named vector of such values that set which expression data to use.
See \code{\link{GeneTargeting}} for specifics and examples -- Seurat and SingleCellExperiment objects deal with these differently, and functionality additions in dittoSeq have led to some minimal divergence from the native methodologies.}

\item{adjustment}{When plotting gene expression (or antibody, or other forms of counts data), should that data be used directly or should it be adjusted to be
\itemize{
\item{"z-score": DEFAULT, centered and scaled to produce a relative-to-mean z-score representation}
\item{NULL: no adjustment, the normal method for all other ditto expression plotting functions}
\item{"relative.to.max": divided by the maximum expression value to give percent of max values between [0,1]}
}}

\item{swap.rownames}{optionally named string or string vector.
For SummarizedExperiment or SingleCellExperiment objects, its value(s) specifies the column name of rowData(object) to be used to identify features instead of rownames(object).
When targeting multiple modalities (alternative experiments), names can be used to specify which level / alternative experiment (use 'main' for the top-level) individual values should be used for.
See \code{\link{GeneTargeting}} for more specifics and examples.}

\item{do.hover}{Logical. Default = \code{FALSE}.
If set to \code{TRUE} (and if there is a "jitter" in \code{plots}): the object will be converted to a plotly object in which underlying data about individual points will be displayed when you hover your cursor over them.}

\item{main}{String which sets the plot title.}

\item{sub}{String which sets the plot subtitle.}

\item{ylab}{String which sets the y axis label.
Default = a combination of the name of the summary function + \code{adjustment} + "expression".
Set to \code{NULL} to remove.}

\item{y.breaks}{Numeric vector, a set of breaks that should be used as major grid lines. c(break1,break2,break3,etc.).}

\item{min, max}{Scalars which control the zoom of the plot.
These inputs set the minimum / maximum values of the data to display.
Default = NA, which allows ggplot to set these limits based on the range of all data being shown.}

\item{xlab}{String which sets the grouping-axis label (=x-axis for box and violin plots, y-axis for ridgeplots).
Set to \code{NULL} to remove.}

\item{x.labels}{String vector, c("label1","label2","label3",...) which overrides the names of groupings.}

\item{x.labels.rotate}{Logical which sets whether the labels should be rotated.
Default: \code{TRUE} for violin and box plots, but \code{FALSE} for ridgeplots.}

\item{x.reorder}{Integer vector. A sequence of numbers, from 1 to the number of groupings, for rearranging the order of x-axis groupings.

Method: Make a first plot without this input.
Then, treating the leftmost grouping as index 1, and the rightmost as index n.
Values of x.reorder should be these indices, but in the order that you would like them rearranged to be.

Recommendation for advanced users: If you find yourself coming back to this input too many times, an alternative solution that can be easier long-term
is to make the target data into a factor, and to put its levels in the desired order: \code{factor(data, levels = c("level1", "level2", ...))}.
\code{\link{metaLevels}} can be used to quickly get the identities that need to be part of this 'levels' input.}

\item{groupings.drop.unused}{Logical. \code{TRUE} by default. If \code{group.by}-data is a factor, factor levels are retained for ordering purposes, but some level(s) can end up with zero cells left after \code{cells.use} subsetting.
By default, we remove them, but you can set this input to \code{FALSE} to keep them.}

\item{color.panel}{String vector which sets the colors to draw from for plot fills.}

\item{colors}{Integer vector, the indexes / order, of colors from color.panel to actually use.
(Provides an alternative to directly modifying \code{color.panel}.)}

\item{theme}{A ggplot theme which will be applied before dittoSeq adjustments.
Default = \code{theme_classic()}.
See \url{https://ggplot2.tidyverse.org/reference/ggtheme.html} for other options and ideas.}

\item{jitter.size}{Scalar which sets the size of the jitter shapes.}

\item{jitter.width}{Scalar that sets the width/spread of the jitter in the x direction. Ignored in ridgeplots.

Note for when \code{color.by} is used to split x-axis groupings into additional bins: ggplot does not shrink jitter widths accordingly, so be sure to do so yourself!
Ideally, needs to be 0.5/num_subgroups.}

\item{jitter.color}{String which sets the color of the jitter shapes}

\item{jitter.position.dodge}{Scalar which adjusts the relative distance between jitter widths when multiple subgroups exist per \code{group.by} grouping (a.k.a. when \code{group.by} and \code{color.by} are not equal).
Similar to \code{boxplot.position.dodge} input & defaults to the value of that input so that BOTH will actually be adjusted when only, say, \code{boxplot.position.dodge = 0.3} is given.}

\item{do.raster}{Logical. When set to \code{TRUE}, rasterizes the jitter plot layer, changing it from individually encoded points to a flattened set of pixels.
This can be useful for editing in external programs (e.g. Illustrator) when there are many thousands of data points.}

\item{raster.dpi}{Number indicating dots/pixels per inch (dpi) to use for rasterization. Default = 300.}

\item{boxplot.width}{Scalar which sets the width/spread of the boxplot in the x direction}

\item{boxplot.color}{String which sets the color of the lines of the boxplot}

\item{boxplot.show.outliers}{Logical, whether outliers should by including in the boxplot.
Default is \code{FALSE} when there is a jitter plotted, \code{TRUE} if there is no jitter.}

\item{boxplot.outlier.size}{Scalar which adjusts the size of points used to mark outliers}

\item{boxplot.fill}{Logical, whether the boxplot should be filled in or not.
Known bug: when boxplot fill is turned off, outliers do not render.}

\item{boxplot.position.dodge}{Scalar which adjusts the relative distance between boxplots when multiple are drawn per grouping (a.k.a. when \code{group.by} and \code{color.by} are not equal).
By default, this input actually controls the value of \code{jitter.position.dodge} unless the \code{jitter} version is provided separately.}

\item{boxplot.lineweight}{Scalar which adjusts the thickness of boxplot lines.}

\item{vlnplot.lineweight}{Scalar which sets the thickness of the line that outlines the violin plots.}

\item{vlnplot.width}{Scalar which sets the width/spread of violin plots in the x direction}

\item{vlnplot.scaling}{String which sets how the widths of the of violin plots are set in relation to each other.
Options are "area", "count", and "width". If the default is not right for your data, I recommend trying "width".
For an explanation of each, see \code{\link{geom_violin}}.}

\item{vlnplot.quantiles}{Single number or numeric vector of values in [0,1] naming quantiles at which to draw a horizontal line within each violin plot. Example: \code{c(0.1, 0.5, 0.9)}}

\item{ridgeplot.lineweight}{Scalar which sets the thickness of the ridgeplot outline.}

\item{ridgeplot.scale}{Scalar which sets the distance/overlap between ridgeplots.
A value of 1 means the tallest density curve just touches the baseline of the next higher one.
Higher numbers lead to greater overlap.  Default = 1.25}

\item{ridgeplot.ymax.expansion}{Scalar which adjusts the minimal space between the top-most grouping and the top of the plot in order to ensure that the curve is not cut off by the plotting grid.
The larger the value, the greater the space requested.
When left as NA, dittoSeq will attempt to determine an ideal value itself based on the number of groups & linear interpolation between these goal posts: 0.6 when g<=3, 0.1 when g==12, and 0.05 when g>=34, where g is the number of groups.}

\item{ridgeplot.shape}{Either "smooth" or "hist", sets whether ridges will be smoothed (the typical, and default) versus rectangular like a histogram.
(Note: as of the time shape "hist" was added, combination of jittered points is not supported by the \code{\link[ggridges]{stat_binline}} that dittoSeq relies on.)}

\item{ridgeplot.bins}{Integer which sets how many chunks to break the x-axis into when \code{ridgeplot.shape = "hist"}.
Overridden by \code{ridgeplot.binwidth} when that input is provided.}

\item{ridgeplot.binwidth}{Integer which sets the width of chunks to break the x-axis into when \code{ridgeplot.shape = "hist"}.
Takes precedence over \code{ridgeplot.bins} when provided.}

\item{add.line}{numeric value(s) where one or multiple line(s) should be added}

\item{line.linetype}{String which sets the type of line for \code{add.line}.
Defaults to "dashed", but any ggplot linetype will work.}

\item{line.color}{String that sets the color(s) of the \code{add.line} line(s)}

\item{split.nrow, split.ncol}{Integers which set the dimensions of faceting/splitting when a single metadata is given to \code{split.by}.}

\item{split.adjust}{A named list which allows extra parameters to be pushed through to the faceting function call.
List elements should be valid inputs to the faceting functions, e.g. `list(scales = "free")`.

For options, when giving 1 metadata to \code{split.by}, see \code{\link[ggplot2]{facet_wrap}},
OR when giving 2 metadatas to \code{split.by}, see \code{\link[ggplot2]{facet_grid}}.}

\item{legend.show}{Logical. Whether the legend should be displayed. Default = \code{TRUE}.}

\item{legend.title}{String or \code{NULL}, sets the title for the main legend which includes colors and data representations.}

\item{data.out}{Logical. When set to \code{TRUE}, changes the output, from the plot alone, to a list containing the plot (\code{p}) and data (\code{data}).}
}
\value{
a ggplot object

Alternatively when \code{data.out = TRUE}, a list containing the plot ("p") and the underlying data as a dataframe ("data").

Alternatively when \code{do.hover = TRUE}, a plotly converted version of the plot where additional data will be displayed when the cursor is hovered over jitter points.
}
\description{
Generates a dittoPlot where data points are genes/metadata summaries, per groups, instead of individual values per cells/samples.
}
\details{
Generally, this function will output a dittoPlot where each data point represents a gene (or metadata) rather than a cell/sample.
Values are the summary (\code{mean} by default) of the values for each gene or metadata requested with \code{vars}, within each group set by \code{group.by}.

To start with, the data for each element of \code{vars} is obtained.
When elements are genes/features, \code{assay} and \code{slot} are utilized to determine which expression data to use,
and \code{adjustment} determines if and how the expression data might be adjusted.
By default, a z-score adjustment is applied to all gene/feature \code{vars}.
Note that this adjustment is applied \emph{before} cells/samples subsetting.

x-axis groupings are then determined using \code{group.by}, and data for each variable is summarized using the \code{summary.fxn}.

Finally, data is plotted with the data representation types in \code{plots}.
}
\section{Plot Customization}{

The \code{plots} argument determines the types of \strong{data representation} that will be generated, as well as their order from back to front.
Options are \code{"jitter"}, \code{"boxplot"}, \code{"vlnplot"}, and \code{"ridgeplot"}.

Each plot type has specific associated options which are controlled by variables that start with their associated string.
For example, all jitter adjustments start with "\code{jitter.}", such as \code{jitter.size} and \code{jitter.width}.

Inclusion of \code{"ridgeplot"} overrides \code{"boxplot"} and \code{"vlnplot"} presence and changes the plot to be horizontal.

Additionally:

\itemize{
\item \strong{Colors can be adjusted} with \code{color.panel}.
\item \strong{Subgroupings:} \code{color.by} can be utilized to split major \code{group.by} groupings into subgroups.
When this is done in y-axis plotting, dittoSeq automatically ensures the centers of all geoms will align,
but users will need to manually adjust \code{jitter.width} to less than 0.5/num_subgroups to avoid overlaps.
There are also three inputs through which one can use to control geom-center placement, but the easiest way to do all at once so is to just adjust \code{vlnplot.width}!
The other two: \code{boxplot.position.dodge}, and \code{jitter.position.dodge}.
\item \strong{Line(s) can be added} at single or multiple value(s) by providing these values to \code{add.line}.
Linetype and color are set with \code{line.linetype}, which is "dashed" by default, and \code{line.color}, which is "black" by default.
\item \strong{Titles and axes labels} can be adjusted with \code{main}, \code{sub}, \code{xlab}, \code{ylab}, and \code{legend.title} arguments.
\item The \strong{legend can be hidden} by setting \code{legend.show = FALSE}.
\item \strong{y-axis zoom and tick marks} can be adjusted using \code{min}, \code{max}, and \code{y.breaks}.
\item \strong{x-axis labels and groupings} can be changed / reordered using \code{x.labels} and \code{x.reorder}, and rotation of these labels can be turned on/off with \code{x.labels.rotate = TRUE/FALSE}.
\item \strong{Shapes used} in conjunction with \code{shape.by} can be adjusted with \code{shape.panel}.
}
}

\examples{
example(importDittoBulk, echo = FALSE)

# Pick a set of genes
genes <- getGenes(myRNA)[1:30]

dittoPlotVarsAcrossGroups(
    myRNA, genes, group.by = "timepoint")

# Color can be controlled separately from grouping with 'color.by'
#   Just note: all groupings must map to a single color.
dittoPlotVarsAcrossGroups(myRNA, genes, "timepoint",
    color.by = "conditions")

# To change it to have the violin plot in the back, a jitter on
#  top of that, and a white boxplot with no fill in front:
dittoPlotVarsAcrossGroups(myRNA, genes, "timepoint",
    plots = c("vlnplot","jitter","boxplot"),
    boxplot.color = "white",
    boxplot.fill = FALSE)

## Data can be summarized in other ways by changing the summary.fxn input.
#  median
dittoPlotVarsAcrossGroups(myRNA, genes, "timepoint",
    summary.fxn = median,
    adjustment = NULL)
#  Percent non-zero expression ( = boring for this fake data)
percent <- function(x) {sum(x!=0)/length(x)}
dittoPlotVarsAcrossGroups(myRNA, genes, "timepoint",
    summary.fxn = percent,
    adjustment = NULL)

# To investigate the identities of outlier genes, we can turn on hovering
# (if the plotly package is available)
if (requireNamespace("plotly", quietly = TRUE)) {
    dittoPlotVarsAcrossGroups(
        myRNA, genes, "timepoint",
        do.hover = TRUE)
}

}
\seealso{
\code{\link{dittoPlot}} and \code{\link{multi_dittoPlot}} for plotting of single or mutliple expression and metadata vars, each as separate plots, on a per cell/sample basis.

\code{\link{dittoDotPlot}} for an alternative representation of per-group summaries of multiple vars where all vars are displayed separately, but still in a single plot.
}
\author{
Daniel Bunis
}
