#-------------------------------------------------------------------------------
#' @title Generate Phosfate Input file
#' 
#' @description It takes as input the `imputedL2fcExtended.txt` results
#' generated by the `artmsAnalysisQuantifications()` function and generates
#' the [Phosfate](http://phosfate.com/) input file (or data.frame)
#' Please, notice that the only species suported by Phosfate is humans.
#' @param inputFile (char) the `imputedL2fcExtended.txt` file name and location
#' @param output_dir (char) Name of the folder to output results 
#' (Default: current directory. Recommended: `phosfate_input`)
#' @param verbose (logical) `TRUE` (default) to show function messages
#' @return Multiple output files (inputs of phosfate)
#' @keywords generate, outputs, files
#' @examples \dontrun{
#' artmsPhosfateOutput(inputFile)
#' }
#' @export
artmsPhosfateOutput <- function(inputFile, 
                                output_dir = ".",
                                verbose = TRUE){

  df <- .artms_checkIfFile(inputFile)
  
  # Check the existance of the columns that are needed.
  checkColumns <- c('Protein', 'PTMsite', 'iLog2FC', 'Comparison')
  
  if (any( !checkColumns %in% colnames(df)) ) {
    stop("One (or many) column names are not found: ",
         sprintf('%s, ',checkColumns))
  }
    
  # create output directory if it doesn't exist
  if(!dir.exists(output_dir)){
    dir.create(output_dir, recursive = TRUE)
  }
  
  conditions <- unique(df$Comparison)
  
  for ( i in seq_len(length(conditions)) ){
    if(verbose) message("+---", i, conditions[i], appendLF = FALSE)
    df.select <- df[which(df$Comparison == conditions[i]),]
    df.out <- df.select[c('Protein','PTMsite','iLog2FC')]
    fileout <- gsub(".txt","", inputFile)
    fileout <- paste0(fileout,"-",conditions[i],".txt")
    fileout <- paste0(output_dir,"/",fileout)
    write.table(df.out, fileout, col.names = FALSE, 
                row.names = FALSE, 
                quote = FALSE, sep = ",")
    if(verbose) message(":", fileout, " is out ")
  }
}

#-------------------------------------------------------------------------------
#' @title Generate PHOTON Input file
#' 
#' @description It takes as input the `imputedL2fcExtended.txt` results
#' generated by the `artmsAnalysisQuantifications()` function and generates
#' the [PHOTON](https://github.com/jdrudolph/photon) input file.
#' Please, notice that the only species suported by PHOTON is humans.
#' @param inputFile (char) the `imputedL2fcExtended.txt` file name and location
#' @param output_dir (char) Name of the folder to output results 
#' (Default: current. Recommended: "photon_input_files" or similar)
#' @param verbose (logical) `TRUE` (default) to show function messages
#' @return Multiple output files (inputs of phosfate)
#' @keywords generate, outputs, files
#' @examples \dontrun{
#' artmsPhotonOutput(inputFile)
#' }
#' @export
artmsPhotonOutput <- function(inputFile,
                              output_dir = ".",
                              verbose = TRUE){
  
  df <- .artms_checkIfFile(inputFile)
  
  # Check the existance of the columns that are needed.
  checkColumns <- c('Gene', 'PTMsite', 'iLog2FC', 'EntrezID')
  
  if (any( !checkColumns %in% colnames(df)) ) {
    stop("One (or many) column names are not found: ",sprintf('%s, ',checkColumns))
  }
  
  # create output directory if it doesn't exist
  if(!dir.exists(output_dir)){
    dir.create(output_dir, recursive = TRUE)
  }
  
  conditions <- unique(df$Comparison)
  
  for ( i in seq_len(length(conditions)) ){
    if(verbose) message("+---", i, conditions[i], appendLF = FALSE)
    df.select <- df[which(df$Comparison == conditions[i]),]
    
    # first, filter by pvalue
    df.select <- df.select[which(df.select$iPvalue < 0.05),]
    
    # Photon takes this input columns: GeneID,Amino.Acid,Position,avg,Symbol
    df.out <- df.select[c('Gene','PTMsite','iLog2FC','EntrezID')]
    df.out$Amino.Acid <- df.select$PTMaa
    df.out <- artmsChangeColumnName(df.out, "EntrezID", "GeneID")
    df.out <- artmsChangeColumnName(df.out, "PTMsite", "Position")
    df.out <- artmsChangeColumnName(df.out, "iLog2FC", "avg")
    df.out <- artmsChangeColumnName(df.out, "Gene", "Symbol")
    
    df.out <- df.out[c('GeneID', 'Amino.Acid', 'Position', 'avg', 'Symbol')]
    if(any(grep(";", df.out$Symbol))){
      if(verbose) message("\t---(-) Removed multiple ids from column 'Protein' ")
      df.out <- df.out[-grep(";", df.out$Symbol),]
    }
    fileout <- gsub(".txt","", inputFile)
    fileout <- paste0("photon.",conditions[i],".csv")
    fileout <- paste0(output_dir,"/",fileout)
    write.table(df.out, fileout, 
                col.names = TRUE, row.names = FALSE, 
                quote = FALSE, sep = ",")
    if(verbose) message(":",fileout," is out ")
  }
}


