#' Identify potential specialized transduction events on contigs classified as
#' Prophage-like
#'
#' Search contigs classified as Prophage-like for dense read coverage outside of
#' the pattern-match borders that may indicate specialized transduction. Returns
#' a list with the first object containing a summary table and the second object
#' containing a list of plots of with associated specialzied transduction search
#' results. If the plot is green, it has been identified as having potential
#' specialized transduction.
#'
#' @param VLPpileup VLP-fraction pileup file generated by mapping sequencing reads
#' from a sample's ultra-purified VLP-fraction mapped to the sample's whole-community
#' metagenome assembly. The pileup file MUST have the following format:
#' * V1: Contig accession
#' * V2: Mapped read coverage values averaged over 100 bp windows
#' * V3: Starting position (bp) of each 100 bp window. Restarts from 0 at the
#' start of each new contig.
#' * V4: Starting position (bp) of each 100 bp window. Does NOT restart at the
#' start of each new contig.
#' @param TrIdentResults Output from `TrIdentClassifier()`
#' @param noReadCov Number of basepairs of zero read coverage encountered before
#'   specialized transduction searching stops. Default is 500. Must be at least
#'   100.
#' @param specTransLength Number of basepairs of non-zero read coverage needed
#'   for specialized transduction to be considered. Default is 2000. Must be
#'   at least 100.
#' @param specificContig Optional, Search a specific contig classified as
#'   Prophage-like ("NODE_1").
#' @param logScale TRUE or FALSE, display VLP-fraction read coverage in log10
#'   scale. Default is FALSE.
#' @param verbose TRUE or FALSE. Print progress messages to console. Default is TRUE.
#' @param SaveFilesTo Provide a path to the directory you wish to save output
#'   to. `specializedTransductionID()` will make a folder within the provided
#'   directory to store results.
#' @return Large list containing two objects
#' @export
#' @examples
#' data("VLPFractionSamplePileup")
#' data("TrIdentSampleOutput")
#'
#' specTransduction <- specializedTransductionID(
#'   VLPpileup = VLPFractionSamplePileup,
#'   TrIdentResults = TrIdentSampleOutput
#' )
#'
#' specTransductionNODE62 <- specializedTransductionID(
#'   VLPpileup = VLPFractionSamplePileup,
#'   TrIdentResults = TrIdentSampleOutput,
#'   specificContig = "NODE_62"
#' )
specializedTransductionID <- function(VLPpileup,
                                      TrIdentResults,
                                      specificContig,
                                      noReadCov = 500,
                                      specTransLength = 2000,
                                      logScale = FALSE,
                                      verbose = TRUE,
                                      SaveFilesTo) {
  ## error catching
  if (noReadCov < 100) {
    stop("noReadCov must be greater than or equal to 100!")
  }
  if (specTransLength < 100) {
    stop("specTransLength must be greater than or equal to 100!")
  }
  ## input validation
  if (abs(VLPpileup[1, 3] - VLPpileup[2, 3]) != 100) {
    stop("VLP pileup file MUST have a windowSize/binsize of 100!")
  }
  specTransInfo <- NULL
  TrIdentResultPatterns <- TrIdentResults[[3]]
  TrIdentResultSumm <- TrIdentResults[[1]]
  windowSize <- TrIdentResults[[5]]
  specTransSumm <- data.frame(matrix(ncol = 5, nrow = 0))
  colnames(specTransSumm) <-
    c(
      "contigName",
      "specTransduc",
      "location",
      "lengthLeft",
      "lengthRight"
    )
  VLPpileup <- pileupFormatter(VLPpileup)
  specificContig <-
    ifelse(missing(specificContig), NA, specificContig)
  specTransCount <- 0
  plots <- vector(mode = "list", length = length(TrIdentResultPatterns))
  J <- 1
  for (i in seq_along(TrIdentResultPatterns)) {
    classification <- TrIdentResultPatterns[[i]][[7]]
    contigName <- TrIdentResultPatterns[[i]][[8]]
    if (is.na(specificContig)) {
      if (classification == "Prophage-like") {
        specTransInfo <-
          specTransductionSearch(
            contigName,
            VLPpileup,
            TrIdentResultPatterns,
            TrIdentResultSumm,
            windowSize,
            i,
            noReadCov,
            specTransLength,
            logScale
          )
        if (specTransInfo[[1]][[2]] == "yes") {
          specTransCount <- specTransCount + 1
        }
        specTransSumm[J, seq_len(5)] <- specTransInfo[[1]]
        plots[[J]] <- specTransInfo[[2]]
        J <- J + 1
      }
    } else if (contigName == specificContig & classification ==
      "Prophage-like") {
      specTransInfo <- specTransductionSearch(
        contigName,
        VLPpileup,
        TrIdentResultPatterns,
        TrIdentResultSumm,
        windowSize,
        i,
        noReadCov,
        specTransLength,
        logScale
      )
      if (specTransInfo[[1]][[2]] == "yes") {
        specTransCount <- specTransCount + 1
      }
      specTransSumm[J, seq_len(5)] <- specTransInfo[[1]]
      plots[[J]] <- specTransInfo[[2]]
      J <- J + 1
    } else if (contigName == specificContig & classification != 
        "Prophage-like") {
        stop("Selected contig is not prophage-like.")
        }
  }
  plots <- plots[!vapply(plots, is.null, logical(1))]
  if (length(plots) == 0){
      stop("No prophage-like events found. If you used the 'specificContig'
parameter, check to see if the contig name is spelled correctly.")
  }
  if (verbose) {
    message(
      specTransCount,
      " contigs have potential specialized transduction"
    )
  }
  if (logScale == FALSE) {
    if (verbose) {
      message(
        "We recommend that you also view the results of this search with
logScale=TRUE"
      )
    }
  }
  names(plots) <- specTransSumm[, 1]
  specTransList <- list(summaryTable = specTransSumm, Plots = plots)
  if (missing(SaveFilesTo) == FALSE) {
    ifelse(!dir.exists(paths = paste0(
      SaveFilesTo,
      "\\TrIdentSpecTransduction"
    )),
    dir.create(paste0(
      SaveFilesTo, "\\TrIdentSpecTransduction"
    )),
    stop(
      "'TrIdentSpecTransduction' folder exists already in the provided
directory"
    )
    )
    lapply(
      names(plots),
      function(X) {
        ggsave(
          filename = paste0(
            SaveFilesTo,
            "\\TrIdentSpecTransduction\\", X, ".png"
          ),
          plot = plots[[X]],
          width = 8,
          height = 4
        )
      }
    )
    write.table(
      specTransSumm,
      file = paste0(
        SaveFilesTo,
        "\\TrIdentSpecTransduction\\SpecTransducSummaryTable.csv"
      ),
      sep = ",",
      row.names = FALSE
    )
    return(specTransList)
  } else {
    return(specTransList)
  }
}
