#' Plot read coverage graphs of contigs classified as Prophage-like, Sloping, or
#' HighCovNoPattern
#'
#' Plot the read coverages of a contig and its associated pattern-match for
#' Prophage-like, Sloping and HighCovNoPattern classifications. Returns a list
#' of ggplot objects.
#'
#' @param VLPpileup VLP-fraction pileup file generated by mapping sequencing reads
#' from a sample's ultra-purified VLP-fraction mapped to the sample's whole-community
#' metagenome assembly. The pileup file MUST have the following format:
#' * V1: Contig accession
#' * V2: Mapped read coverage values averaged over 100 bp windows
#' * V3: Starting position (bp) of each 100 bp window. Restarts from 0 at the
#' start of each new contig.
#' * V4: Starting position (bp) of each 100 bp window. Does NOT restart at the
#' start of each new contig.
#' @param WCpileup A whole-community pileup file generated by mapping sequencing reads
#' from a sample's whole-community mapped to the sample's whole-community metagenome assembly.
#' The pileup file MUST have the following format:
#' * V1: Contig accession
#' * V2: Mapped read coverage values averaged over 100 bp windows
#' * V3: Starting position (bp) of each 100 bp window. Restarts from 0 at the
#' start of each new contig.
#' * V4: Starting position (bp) of each 100 bp window. Does NOT restart at the
#' start of each new contig.
#' @param TrIdentResults Output from `TrIdentClassifier()`.
#' @param onlyPlot Optional, use to 'only plot' the contigs classified as either 
#' "Prophage-like", "Sloping", or "HighCovNoPattern".
#' @param logScale TRUE or FALSE, display VLP-fraction read coverage in log10
#'   scale. Default is FALSE.
#' @param saveFilesTo Optional, Provide a path to the directory you wish to save
#'   output to. A folder will be made within the provided directory to store
#'   results. 
#' @return Large list containing ggplot objects
#' @export
#' @examples
#' data("VLPFractionSamplePileup")
#' data("WholeCommunitySamplePileup")
#' data("TrIdentSampleOutput")
#'
#' patternMatches <- plotTrIdentResults(
#'   VLPpileup = VLPFractionSamplePileup,
#'   WCpileup = WholeCommunitySamplePileup,
#'   TrIdentResults = TrIdentSampleOutput
#' )
plotTrIdentResults <- function(VLPpileup,
                               WCpileup,
                               TrIdentResults,
                               onlyPlot,
                               logScale = FALSE,
                               saveFilesTo) {
  ## input validation
  if (nrow(VLPpileup) != nrow(WCpileup)) {
    stop("VLP and WC pileup files have differing row numbers")
  }
  if (abs(VLPpileup[1, 3] - VLPpileup[2, 3]) != 100 |
    abs(WCpileup[1, 3] - WCpileup[2, 3]) != 100) {
    stop("pileup files MUST have a windowSize/binsize of 100!")
  }
  if (all(VLPpileup[, 1] == WCpileup[, 1]) == FALSE) {
    stop("The first column of the VLP and WC pileup file should be identical if
         mapping was performed correctly...")
  }
  position <- coverage <- NULL
  windowSize <- TrIdentResults[[5]]
  summaryTable <- TrIdentResults[[2]]
  summaryTable$rowIndexes <- seq(1,nrow(summaryTable))
  classifList <- TrIdentResults[[3]]
  VLPpileup <- pileupFormatter(VLPpileup)
  WCpileup <- pileupFormatter(WCpileup)
  summaryTable <- 
      if (missing(onlyPlot)){
          summaryTable
          } else { 
          summaryTable[grep(onlyPlot, summaryTable[,2], 
                            ignore.case=TRUE),]}
  if (nrow(summaryTable) == 0){
      stop("There are no contigs classified as the provided `onlyPlot` 
    parameter. Is the classification name spelled correctly? The options 
    are either 'Prophage-like', 'Sloping', or 'HighCovNoPattern'.")
  }
  plots <- lapply(seq_len(nrow(summaryTable)), function(i) {
    rowIndex <- summaryTable$rowIndexes[[i]]
    contigName <- summaryTable[i,1]
    patternMatchInfo <-
        summaryTable[which(summaryTable[, 1] == contigName), ]
    classification <- patternMatchInfo[, 2]
    matchLength <- patternMatchInfo[, 5]
    viralSubset <-
      changeWindowSize(
        VLPpileup[which(VLPpileup[, 1] == contigName), ],
        windowSize
      )
    microbialSubset <-
      changeWindowSize(
        WCpileup[which(WCpileup[, 1] == contigName), ],
        windowSize
      )
    if (logScale) {
        viralSubset$logcoverage <- abs(log10(viralSubset[, 2]))
        viralSubset[viralSubset == Inf] <- 0
        microbialSubset$logcoverage <- abs(log10(microbialSubset[, 2]))
        microbialSubset[microbialSubset == Inf] <- 0
        coverageTypeViral <- viralSubset$logcoverage
        coverageTypeMicrobial <- microbialSubset$logcoverage
        pattern <- rep(0, nrow(viralSubset))
    } else {
        coverageTypeViral <- viralSubset$coverage
        coverageTypeMicrobial <- microbialSubset$coverage
        pattern <-
            patternBuilder(viralSubset, classifList, classification, rowIndex)
    }
    viralSubset <- cbind.data.frame(viralSubset, pattern)
    if (classification == "Sloping") {
      subtitleInfo <- paste(
        "Slope:",
        patternMatchInfo[10]
      )
    } else if (classification == "HighCovNoPattern") {
      subtitleInfo <- paste("VLP:WC ratio:", patternMatchInfo[4])
    } else if (classification == "Prophage-like") {
      if (is.na(patternMatchInfo[8])) {
        subtitleInfo <- NULL
      } else if (patternMatchInfo[8] == "Elevated") {
        subtitleInfo <- "Active/highly abundant Prophage-like element"
      } else if (patternMatchInfo[8] == "Depressed") {
        subtitleInfo <- "Not homogenously integrated Prophage-like element"
      } else {
        subtitleInfo <- NULL
      }
    }
    wholecomm_plot <-
      ggplot(data = microbialSubset, aes(x = position, y = coverageTypeMicrobial)) +
      geom_area(fill = "deepskyblue3") +
      labs(
        title = paste(contigName, "Classification:", classification),
        subtitle = paste(
          "Match size (bp):", matchLength,
          subtitleInfo
        ),
        x = " ",
        y = paste(
            "Whole-community \n read coverage",
            ifelse(logScale,
                   "\n (Log10)", ""))
      ) +
      scale_x_continuous(expand = c(0, 0)) +
      theme(
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.background = element_blank(),
        axis.line = element_line(colour = "black"),
        text = element_text(size = 14),
        axis.text = element_text(size = 11),
        plot.subtitle = element_text(size = 12),
        plot.title = element_text(size = 14),
        plot.margin = margin(
          t = 0,
          r = 6,
          b = 0,
          l = 2
        )
      )
    Overlay_plot <-
      ggplot(data = viralSubset, aes(x = position, y = coverageTypeViral)) +
      geom_area(fill = "deepskyblue3") +
      geom_line(aes(y = pattern),
        color = "black",
        linewidth = 1
      ) +
      labs(
        x = "Contig position (bp)",
        y = paste(
            "VLP-fraction \n read coverage",
            ifelse(logScale,
                   "\n (Log10)", ""))
      ) +
      scale_x_continuous(expand = c(0, 0)) +
      theme(
        panel.grid.major = element_blank(),
        panel.grid.minor = element_blank(),
        panel.background = element_blank(),
        axis.line = element_line(colour = "black"),
        text = element_text(size = 14),
        axis.text = element_text(size = 11),
        plot.margin = margin(
          t = 0,
          r = 6,
          b = 0,
          l = 2
        )
      )
    combined_plot <- (wholecomm_plot / Overlay_plot)
    combined_plot
  })
  plots <- Filter(Negate(is.null), plots)
  contigNames <- vapply(seq_len(nrow(summaryTable)), function(i) {
      contigName <- summaryTable[i,1]
      viralSubset <-
          changeWindowSize(
              VLPpileup[which(VLPpileup[, 1] == contigName), ],
              windowSize
          )
      contigName
  }, character(1))
  contigNames <- (contigNames[!vapply(contigNames, is.null, logical(1))])
  names(plots) <- contigNames
  if (missing(saveFilesTo) == FALSE) {
    ifelse(!dir.exists(paths = paste0(saveFilesTo, "\\TrIdentOutput")),
      dir.create(paste0(saveFilesTo, "\\TrIdentOutput")),
      stop(
        "'TrIdentOutput' folder exists already in the provided directory"
      )
    )
    lapply(
      names(plots),
      function(X) {
        ggsave(
          filename = paste0(
            saveFilesTo,
            "\\TrIdentOutput\\", X, ".png"
          ),
          plot = plots[[X]],
          width = 8,
          height = 4
        )
      }
    )
    return(plots)
  } else {
    return(plots)
  }
}

#' Pattern-builder
#'
#' Builds the pattern (vector) associated with the best pattern-match' for each
#' contig classified as Prophage-like, Sloping, or HighCovNoPattern.
#'
#' @param viralSubset A subset of the read coverage pileup that pertains only to
#'   the contig currently being assessed
#' @param classifList A list containing pattern match information associated
#'   with all classified contigs.
#' @param rowIndex The list index associated with each contig's pattern-match
#'   information
#' @param classification The contig's classification assigned by the
#'   TrIdentClassifier function
#' @return Vector
#' @keywords internal
patternBuilder <-
  function(viralSubset,
           classifList,
           classification,
           rowIndex) {
    minReadCov <- classifList[[rowIndex]][[2]]
    maxReadCov <- classifList[[rowIndex]][[3]]
    slopingCovSteps <- classifList[[rowIndex]][[4]]
    startPos <- classifList[[rowIndex]][[5]]
    endPos <- classifList[[rowIndex]][[6]]
    if (classification == "Prophage-like") {
      if (startPos == 1) {
        pattern <- c(
          rep(maxReadCov, endPos),
          rep(
            minReadCov,
            (nrow(viralSubset) - endPos)
          )
        )
      } else if (endPos == nrow(viralSubset)) {
        pattern <- c(
          rep(
            minReadCov,
            startPos
          ),
          rep(
            maxReadCov,
            (nrow(viralSubset) -
              startPos)
          )
        )
      } else {
        blockRegion <- endPos - startPos
        pattern <-
          c(
            rep(minReadCov, startPos),
            rep(maxReadCov, blockRegion),
            rep(
              minReadCov,
              (
                nrow(viralSubset) - (blockRegion + startPos)
              )
            )
          )
      }
    } else if (classification == "Sloping") {
      if (startPos == 1 &
        endPos == nrow(viralSubset) &
        slopingCovSteps < 0) {
        pattern <- seq(maxReadCov, minReadCov, slopingCovSteps)
      } else if (startPos == 1 &
        endPos == nrow(viralSubset) &
        slopingCovSteps > 0) {
        pattern <- seq(minReadCov, maxReadCov, slopingCovSteps)
      } else if (startPos != 1) {
        pattern <- c(
          rep(
            min(viralSubset[, 2]),
            startPos - 1
          ),
          seq(
            maxReadCov,
            minReadCov,
            slopingCovSteps
          )
        )
      } else if (endPos != nrow(viralSubset)) {
        pattern <- c(
          seq(
            minReadCov,
            maxReadCov,
            slopingCovSteps
          ),
          rep(
            min(viralSubset[, 2]),
            (nrow(viralSubset) -
              endPos)
          )
        )
      }
      ## for NoPattern, min read cov = med read cov and
      ## max read cov = contig length
    } else if (classification == "HighCovNoPattern") {
      pattern <- rep(
        minReadCov,
        maxReadCov
      )
    }
    return(pattern)
  }
